#!/bin/sh

# Copyright (C) 2023-2025 Internet Systems Consortium, Inc. ("ISC")
#
# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.

# Exit with error if commands exit with non-zero and if undefined variables are
# used.
set -eu

# shellcheck disable=SC2034
# SC2034: ... appears unused. Verify use (or export if used externally).
prefix="/usr"

# Include utilities based on location of this script. Check for sources first,
# so that the unexpected situations with weird paths fall on the default
# case of installed.
script_path=$(cd "$(dirname "${0}")" && pwd)
if test "${script_path}" = "/build/kea/src/build/src/share/database/scripts/mysql"; then
    # shellcheck source=./src/bin/admin/admin-utils.sh.in
    . "/build/kea/src/build/src/bin/admin/admin-utils.sh"
else
    # shellcheck source=./src/bin/admin/admin-utils.sh.in
    . "${prefix}/share/kea/scripts/admin-utils.sh"
fi

# Check version.
version=$(mysql_version "${@}")
if test "${version}" != "21.0"; then
    printf 'This script upgrades 21.0 to 22.0. '
    printf 'Reported version is %s. Skipping upgrade.\n' "${version}"
    exit 0
fi

# Get the schema name from database argument. We need this to
# query information_schema for the right database.
for arg in "${@}"
do
    if ! printf '%s' "${arg}" | grep -Eq -- '^--'
    then
        schema="$arg"
        break
    fi
done

# Make sure we have the schema.
if [ -z "$schema" ]
then
    printf "Could not find database schema name in cmd line args: %s\n" "${*}"
    exit 255
fi

mysql "$@" <<EOF

-- This line starts the schema upgrade to version 22.0.

-- Turn off Mariadb default/on-update for expire column
ALTER TABLE lease4 MODIFY expire timestamp NULL;
ALTER TABLE lease6 MODIFY expire timestamp NULL;

SET @disable_audit = 1;

DELETE FROM dhcp4_global_parameter WHERE name='dhcp-ddns.override-no-update';
DELETE FROM dhcp4_global_parameter WHERE name='dhcp-ddns.override-client-update';
DELETE FROM dhcp4_global_parameter WHERE name='dhcp-ddns.replace-client-name';
DELETE FROM dhcp4_global_parameter WHERE name='dhcp-ddns.generated-prefix';
DELETE FROM dhcp4_global_parameter WHERE name='dhcp-ddns.qualifying-suffix';
DELETE FROM dhcp4_global_parameter WHERE name='dhcp-ddns.hostname-char-set';
DELETE FROM dhcp4_global_parameter WHERE name='dhcp-ddns.hostname-char-replacement';

DELETE FROM dhcp6_global_parameter WHERE name='dhcp-ddns.override-no-update';
DELETE FROM dhcp6_global_parameter WHERE name='dhcp-ddns.override-client-update';
DELETE FROM dhcp6_global_parameter WHERE name='dhcp-ddns.replace-client-name';
DELETE FROM dhcp6_global_parameter WHERE name='dhcp-ddns.generated-prefix';
DELETE FROM dhcp6_global_parameter WHERE name='dhcp-ddns.qualifying-suffix';
DELETE FROM dhcp6_global_parameter WHERE name='dhcp-ddns.hostname-char-set';
DELETE FROM dhcp6_global_parameter WHERE name='dhcp-ddns.hostname-char-replacement';

UPDATE dhcp4_global_parameter SET name='reservations-global', value='1', parameter_type=2 WHERE name='reservation-mode' AND value='global';
UPDATE dhcp4_global_parameter SET name='reservations-in-subnet', value='1', parameter_type=2 WHERE name='reservation-mode' AND value='all';
UPDATE dhcp4_global_parameter SET name='reservations-in-subnet', value='0', parameter_type=2 WHERE name='reservation-mode' AND value='disabled';
UPDATE dhcp4_global_parameter SET name='reservations-in-subnet', value='0', parameter_type=2 WHERE name='reservation-mode' AND value='off';
UPDATE dhcp4_global_parameter SET name='reservations-out-of-pool', value='1', parameter_type=2 WHERE name='reservation-mode' AND value='out-of-pool';

UPDATE dhcp6_global_parameter SET name='reservations-global', value='1', parameter_type=2 WHERE name='reservation-mode' AND value='global';
UPDATE dhcp6_global_parameter SET name='reservations-in-subnet', value='1', parameter_type=2 WHERE name='reservation-mode' AND value='all';
UPDATE dhcp6_global_parameter SET name='reservations-in-subnet', value='0', parameter_type=2 WHERE name='reservation-mode' AND value='disabled';
UPDATE dhcp6_global_parameter SET name='reservations-in-subnet', value='0', parameter_type=2 WHERE name='reservation-mode' AND value='off';
UPDATE dhcp6_global_parameter SET name='reservations-out-of-pool', value='1', parameter_type=2 WHERE name='reservation-mode' AND value='out-of-pool';

SET @disable_audit = 0;

-- Update the schema version number.
UPDATE schema_version
    SET version = '22', minor = '0';

-- This line concludes the schema upgrade to version 22.0.
EOF
