﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/securityhub/SecurityHub_EXPORTS.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/securityhub/model/StringFilter.h>
#include <aws/securityhub/model/DateFilter.h>
#include <aws/securityhub/model/NumberFilter.h>
#include <aws/securityhub/model/MapFilter.h>
#include <aws/securityhub/model/IpFilter.h>
#include <aws/securityhub/model/BooleanFilter.h>
#include <utility>

namespace Aws
{
namespace Utils
{
namespace Json
{
  class JsonValue;
  class JsonView;
} // namespace Json
} // namespace Utils
namespace SecurityHub
{
namespace Model
{

  /**
   * <p>A collection of filters that are applied to all active findings aggregated by
   * Security Hub.</p> <p>You can filter by up to ten finding attributes. For each
   * attribute, you can provide up to 20 filter values.</p><p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/securityhub-2018-10-26/AwsSecurityFindingFilters">AWS
   * API Reference</a></p>
   */
  class AwsSecurityFindingFilters
  {
  public:
    AWS_SECURITYHUB_API AwsSecurityFindingFilters() = default;
    AWS_SECURITYHUB_API AwsSecurityFindingFilters(Aws::Utils::Json::JsonView jsonValue);
    AWS_SECURITYHUB_API AwsSecurityFindingFilters& operator=(Aws::Utils::Json::JsonView jsonValue);
    AWS_SECURITYHUB_API Aws::Utils::Json::JsonValue Jsonize() const;


    ///@{
    /**
     * <p>The ARN generated by Security Hub that uniquely identifies a third-party
     * company (security findings provider) after this provider's product (solution
     * that generates findings) is registered with Security Hub.</p>
     */
    inline const Aws::Vector<StringFilter>& GetProductArn() const { return m_productArn; }
    inline bool ProductArnHasBeenSet() const { return m_productArnHasBeenSet; }
    template<typename ProductArnT = Aws::Vector<StringFilter>>
    void SetProductArn(ProductArnT&& value) { m_productArnHasBeenSet = true; m_productArn = std::forward<ProductArnT>(value); }
    template<typename ProductArnT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithProductArn(ProductArnT&& value) { SetProductArn(std::forward<ProductArnT>(value)); return *this;}
    template<typename ProductArnT = StringFilter>
    AwsSecurityFindingFilters& AddProductArn(ProductArnT&& value) { m_productArnHasBeenSet = true; m_productArn.emplace_back(std::forward<ProductArnT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The Amazon Web Services account ID in which a finding is generated.</p>
     */
    inline const Aws::Vector<StringFilter>& GetAwsAccountId() const { return m_awsAccountId; }
    inline bool AwsAccountIdHasBeenSet() const { return m_awsAccountIdHasBeenSet; }
    template<typename AwsAccountIdT = Aws::Vector<StringFilter>>
    void SetAwsAccountId(AwsAccountIdT&& value) { m_awsAccountIdHasBeenSet = true; m_awsAccountId = std::forward<AwsAccountIdT>(value); }
    template<typename AwsAccountIdT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithAwsAccountId(AwsAccountIdT&& value) { SetAwsAccountId(std::forward<AwsAccountIdT>(value)); return *this;}
    template<typename AwsAccountIdT = StringFilter>
    AwsSecurityFindingFilters& AddAwsAccountId(AwsAccountIdT&& value) { m_awsAccountIdHasBeenSet = true; m_awsAccountId.emplace_back(std::forward<AwsAccountIdT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The security findings provider-specific identifier for a finding.</p>
     */
    inline const Aws::Vector<StringFilter>& GetId() const { return m_id; }
    inline bool IdHasBeenSet() const { return m_idHasBeenSet; }
    template<typename IdT = Aws::Vector<StringFilter>>
    void SetId(IdT&& value) { m_idHasBeenSet = true; m_id = std::forward<IdT>(value); }
    template<typename IdT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithId(IdT&& value) { SetId(std::forward<IdT>(value)); return *this;}
    template<typename IdT = StringFilter>
    AwsSecurityFindingFilters& AddId(IdT&& value) { m_idHasBeenSet = true; m_id.emplace_back(std::forward<IdT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The identifier for the solution-specific component (a discrete unit of logic)
     * that generated a finding. In various security findings providers' solutions,
     * this generator can be called a rule, a check, a detector, a plugin, etc.</p>
     */
    inline const Aws::Vector<StringFilter>& GetGeneratorId() const { return m_generatorId; }
    inline bool GeneratorIdHasBeenSet() const { return m_generatorIdHasBeenSet; }
    template<typename GeneratorIdT = Aws::Vector<StringFilter>>
    void SetGeneratorId(GeneratorIdT&& value) { m_generatorIdHasBeenSet = true; m_generatorId = std::forward<GeneratorIdT>(value); }
    template<typename GeneratorIdT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithGeneratorId(GeneratorIdT&& value) { SetGeneratorId(std::forward<GeneratorIdT>(value)); return *this;}
    template<typename GeneratorIdT = StringFilter>
    AwsSecurityFindingFilters& AddGeneratorId(GeneratorIdT&& value) { m_generatorIdHasBeenSet = true; m_generatorId.emplace_back(std::forward<GeneratorIdT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The Region from which the finding was generated.</p>
     */
    inline const Aws::Vector<StringFilter>& GetRegion() const { return m_region; }
    inline bool RegionHasBeenSet() const { return m_regionHasBeenSet; }
    template<typename RegionT = Aws::Vector<StringFilter>>
    void SetRegion(RegionT&& value) { m_regionHasBeenSet = true; m_region = std::forward<RegionT>(value); }
    template<typename RegionT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithRegion(RegionT&& value) { SetRegion(std::forward<RegionT>(value)); return *this;}
    template<typename RegionT = StringFilter>
    AwsSecurityFindingFilters& AddRegion(RegionT&& value) { m_regionHasBeenSet = true; m_region.emplace_back(std::forward<RegionT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>A finding type in the format of <code>namespace/category/classifier</code>
     * that classifies a finding.</p>
     */
    inline const Aws::Vector<StringFilter>& GetType() const { return m_type; }
    inline bool TypeHasBeenSet() const { return m_typeHasBeenSet; }
    template<typename TypeT = Aws::Vector<StringFilter>>
    void SetType(TypeT&& value) { m_typeHasBeenSet = true; m_type = std::forward<TypeT>(value); }
    template<typename TypeT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithType(TypeT&& value) { SetType(std::forward<TypeT>(value)); return *this;}
    template<typename TypeT = StringFilter>
    AwsSecurityFindingFilters& AddType(TypeT&& value) { m_typeHasBeenSet = true; m_type.emplace_back(std::forward<TypeT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>A timestamp that indicates when the security findings provider first observed
     * the potential security issue that a finding captured.</p> <p>For more
     * information about the validation and formatting of timestamp fields in Security
     * Hub, see <a
     * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/Welcome.html#timestamps">Timestamps</a>.</p>
     */
    inline const Aws::Vector<DateFilter>& GetFirstObservedAt() const { return m_firstObservedAt; }
    inline bool FirstObservedAtHasBeenSet() const { return m_firstObservedAtHasBeenSet; }
    template<typename FirstObservedAtT = Aws::Vector<DateFilter>>
    void SetFirstObservedAt(FirstObservedAtT&& value) { m_firstObservedAtHasBeenSet = true; m_firstObservedAt = std::forward<FirstObservedAtT>(value); }
    template<typename FirstObservedAtT = Aws::Vector<DateFilter>>
    AwsSecurityFindingFilters& WithFirstObservedAt(FirstObservedAtT&& value) { SetFirstObservedAt(std::forward<FirstObservedAtT>(value)); return *this;}
    template<typename FirstObservedAtT = DateFilter>
    AwsSecurityFindingFilters& AddFirstObservedAt(FirstObservedAtT&& value) { m_firstObservedAtHasBeenSet = true; m_firstObservedAt.emplace_back(std::forward<FirstObservedAtT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>A timestamp that indicates when the security findings provider most recently
     * observed a change in the resource that is involved in the finding.</p> <p>For
     * more information about the validation and formatting of timestamp fields in
     * Security Hub, see <a
     * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/Welcome.html#timestamps">Timestamps</a>.</p>
     */
    inline const Aws::Vector<DateFilter>& GetLastObservedAt() const { return m_lastObservedAt; }
    inline bool LastObservedAtHasBeenSet() const { return m_lastObservedAtHasBeenSet; }
    template<typename LastObservedAtT = Aws::Vector<DateFilter>>
    void SetLastObservedAt(LastObservedAtT&& value) { m_lastObservedAtHasBeenSet = true; m_lastObservedAt = std::forward<LastObservedAtT>(value); }
    template<typename LastObservedAtT = Aws::Vector<DateFilter>>
    AwsSecurityFindingFilters& WithLastObservedAt(LastObservedAtT&& value) { SetLastObservedAt(std::forward<LastObservedAtT>(value)); return *this;}
    template<typename LastObservedAtT = DateFilter>
    AwsSecurityFindingFilters& AddLastObservedAt(LastObservedAtT&& value) { m_lastObservedAtHasBeenSet = true; m_lastObservedAt.emplace_back(std::forward<LastObservedAtT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>A timestamp that indicates when the security findings provider created the
     * potential security issue that a finding reflects.</p> <p>For more information
     * about the validation and formatting of timestamp fields in Security Hub, see <a
     * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/Welcome.html#timestamps">Timestamps</a>.</p>
     */
    inline const Aws::Vector<DateFilter>& GetCreatedAt() const { return m_createdAt; }
    inline bool CreatedAtHasBeenSet() const { return m_createdAtHasBeenSet; }
    template<typename CreatedAtT = Aws::Vector<DateFilter>>
    void SetCreatedAt(CreatedAtT&& value) { m_createdAtHasBeenSet = true; m_createdAt = std::forward<CreatedAtT>(value); }
    template<typename CreatedAtT = Aws::Vector<DateFilter>>
    AwsSecurityFindingFilters& WithCreatedAt(CreatedAtT&& value) { SetCreatedAt(std::forward<CreatedAtT>(value)); return *this;}
    template<typename CreatedAtT = DateFilter>
    AwsSecurityFindingFilters& AddCreatedAt(CreatedAtT&& value) { m_createdAtHasBeenSet = true; m_createdAt.emplace_back(std::forward<CreatedAtT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>A timestamp that indicates when the security findings provider last updated
     * the finding record.</p> <p>For more information about the validation and
     * formatting of timestamp fields in Security Hub, see <a
     * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/Welcome.html#timestamps">Timestamps</a>.</p>
     */
    inline const Aws::Vector<DateFilter>& GetUpdatedAt() const { return m_updatedAt; }
    inline bool UpdatedAtHasBeenSet() const { return m_updatedAtHasBeenSet; }
    template<typename UpdatedAtT = Aws::Vector<DateFilter>>
    void SetUpdatedAt(UpdatedAtT&& value) { m_updatedAtHasBeenSet = true; m_updatedAt = std::forward<UpdatedAtT>(value); }
    template<typename UpdatedAtT = Aws::Vector<DateFilter>>
    AwsSecurityFindingFilters& WithUpdatedAt(UpdatedAtT&& value) { SetUpdatedAt(std::forward<UpdatedAtT>(value)); return *this;}
    template<typename UpdatedAtT = DateFilter>
    AwsSecurityFindingFilters& AddUpdatedAt(UpdatedAtT&& value) { m_updatedAtHasBeenSet = true; m_updatedAt.emplace_back(std::forward<UpdatedAtT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The label of a finding's severity.</p>
     */
    inline const Aws::Vector<StringFilter>& GetSeverityLabel() const { return m_severityLabel; }
    inline bool SeverityLabelHasBeenSet() const { return m_severityLabelHasBeenSet; }
    template<typename SeverityLabelT = Aws::Vector<StringFilter>>
    void SetSeverityLabel(SeverityLabelT&& value) { m_severityLabelHasBeenSet = true; m_severityLabel = std::forward<SeverityLabelT>(value); }
    template<typename SeverityLabelT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithSeverityLabel(SeverityLabelT&& value) { SetSeverityLabel(std::forward<SeverityLabelT>(value)); return *this;}
    template<typename SeverityLabelT = StringFilter>
    AwsSecurityFindingFilters& AddSeverityLabel(SeverityLabelT&& value) { m_severityLabelHasBeenSet = true; m_severityLabel.emplace_back(std::forward<SeverityLabelT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>A finding's confidence. Confidence is defined as the likelihood that a
     * finding accurately identifies the behavior or issue that it was intended to
     * identify.</p> <p>Confidence is scored on a 0-100 basis using a ratio scale,
     * where 0 means zero percent confidence and 100 means 100 percent confidence.</p>
     */
    inline const Aws::Vector<NumberFilter>& GetConfidence() const { return m_confidence; }
    inline bool ConfidenceHasBeenSet() const { return m_confidenceHasBeenSet; }
    template<typename ConfidenceT = Aws::Vector<NumberFilter>>
    void SetConfidence(ConfidenceT&& value) { m_confidenceHasBeenSet = true; m_confidence = std::forward<ConfidenceT>(value); }
    template<typename ConfidenceT = Aws::Vector<NumberFilter>>
    AwsSecurityFindingFilters& WithConfidence(ConfidenceT&& value) { SetConfidence(std::forward<ConfidenceT>(value)); return *this;}
    template<typename ConfidenceT = NumberFilter>
    AwsSecurityFindingFilters& AddConfidence(ConfidenceT&& value) { m_confidenceHasBeenSet = true; m_confidence.emplace_back(std::forward<ConfidenceT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The level of importance assigned to the resources associated with the
     * finding.</p> <p>A score of 0 means that the underlying resources have no
     * criticality, and a score of 100 is reserved for the most critical resources.</p>
     */
    inline const Aws::Vector<NumberFilter>& GetCriticality() const { return m_criticality; }
    inline bool CriticalityHasBeenSet() const { return m_criticalityHasBeenSet; }
    template<typename CriticalityT = Aws::Vector<NumberFilter>>
    void SetCriticality(CriticalityT&& value) { m_criticalityHasBeenSet = true; m_criticality = std::forward<CriticalityT>(value); }
    template<typename CriticalityT = Aws::Vector<NumberFilter>>
    AwsSecurityFindingFilters& WithCriticality(CriticalityT&& value) { SetCriticality(std::forward<CriticalityT>(value)); return *this;}
    template<typename CriticalityT = NumberFilter>
    AwsSecurityFindingFilters& AddCriticality(CriticalityT&& value) { m_criticalityHasBeenSet = true; m_criticality.emplace_back(std::forward<CriticalityT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>A finding's title.</p>
     */
    inline const Aws::Vector<StringFilter>& GetTitle() const { return m_title; }
    inline bool TitleHasBeenSet() const { return m_titleHasBeenSet; }
    template<typename TitleT = Aws::Vector<StringFilter>>
    void SetTitle(TitleT&& value) { m_titleHasBeenSet = true; m_title = std::forward<TitleT>(value); }
    template<typename TitleT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithTitle(TitleT&& value) { SetTitle(std::forward<TitleT>(value)); return *this;}
    template<typename TitleT = StringFilter>
    AwsSecurityFindingFilters& AddTitle(TitleT&& value) { m_titleHasBeenSet = true; m_title.emplace_back(std::forward<TitleT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>A finding's description.</p>
     */
    inline const Aws::Vector<StringFilter>& GetDescription() const { return m_description; }
    inline bool DescriptionHasBeenSet() const { return m_descriptionHasBeenSet; }
    template<typename DescriptionT = Aws::Vector<StringFilter>>
    void SetDescription(DescriptionT&& value) { m_descriptionHasBeenSet = true; m_description = std::forward<DescriptionT>(value); }
    template<typename DescriptionT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithDescription(DescriptionT&& value) { SetDescription(std::forward<DescriptionT>(value)); return *this;}
    template<typename DescriptionT = StringFilter>
    AwsSecurityFindingFilters& AddDescription(DescriptionT&& value) { m_descriptionHasBeenSet = true; m_description.emplace_back(std::forward<DescriptionT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The recommendation of what to do about the issue described in a finding.</p>
     */
    inline const Aws::Vector<StringFilter>& GetRecommendationText() const { return m_recommendationText; }
    inline bool RecommendationTextHasBeenSet() const { return m_recommendationTextHasBeenSet; }
    template<typename RecommendationTextT = Aws::Vector<StringFilter>>
    void SetRecommendationText(RecommendationTextT&& value) { m_recommendationTextHasBeenSet = true; m_recommendationText = std::forward<RecommendationTextT>(value); }
    template<typename RecommendationTextT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithRecommendationText(RecommendationTextT&& value) { SetRecommendationText(std::forward<RecommendationTextT>(value)); return *this;}
    template<typename RecommendationTextT = StringFilter>
    AwsSecurityFindingFilters& AddRecommendationText(RecommendationTextT&& value) { m_recommendationTextHasBeenSet = true; m_recommendationText.emplace_back(std::forward<RecommendationTextT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>A URL that links to a page about the current finding in the security findings
     * provider's solution.</p>
     */
    inline const Aws::Vector<StringFilter>& GetSourceUrl() const { return m_sourceUrl; }
    inline bool SourceUrlHasBeenSet() const { return m_sourceUrlHasBeenSet; }
    template<typename SourceUrlT = Aws::Vector<StringFilter>>
    void SetSourceUrl(SourceUrlT&& value) { m_sourceUrlHasBeenSet = true; m_sourceUrl = std::forward<SourceUrlT>(value); }
    template<typename SourceUrlT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithSourceUrl(SourceUrlT&& value) { SetSourceUrl(std::forward<SourceUrlT>(value)); return *this;}
    template<typename SourceUrlT = StringFilter>
    AwsSecurityFindingFilters& AddSourceUrl(SourceUrlT&& value) { m_sourceUrlHasBeenSet = true; m_sourceUrl.emplace_back(std::forward<SourceUrlT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>A data type where security findings providers can include additional
     * solution-specific details that aren't part of the defined
     * <code>AwsSecurityFinding</code> format.</p>
     */
    inline const Aws::Vector<MapFilter>& GetProductFields() const { return m_productFields; }
    inline bool ProductFieldsHasBeenSet() const { return m_productFieldsHasBeenSet; }
    template<typename ProductFieldsT = Aws::Vector<MapFilter>>
    void SetProductFields(ProductFieldsT&& value) { m_productFieldsHasBeenSet = true; m_productFields = std::forward<ProductFieldsT>(value); }
    template<typename ProductFieldsT = Aws::Vector<MapFilter>>
    AwsSecurityFindingFilters& WithProductFields(ProductFieldsT&& value) { SetProductFields(std::forward<ProductFieldsT>(value)); return *this;}
    template<typename ProductFieldsT = MapFilter>
    AwsSecurityFindingFilters& AddProductFields(ProductFieldsT&& value) { m_productFieldsHasBeenSet = true; m_productFields.emplace_back(std::forward<ProductFieldsT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The name of the solution (product) that generates findings.</p>
     */
    inline const Aws::Vector<StringFilter>& GetProductName() const { return m_productName; }
    inline bool ProductNameHasBeenSet() const { return m_productNameHasBeenSet; }
    template<typename ProductNameT = Aws::Vector<StringFilter>>
    void SetProductName(ProductNameT&& value) { m_productNameHasBeenSet = true; m_productName = std::forward<ProductNameT>(value); }
    template<typename ProductNameT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithProductName(ProductNameT&& value) { SetProductName(std::forward<ProductNameT>(value)); return *this;}
    template<typename ProductNameT = StringFilter>
    AwsSecurityFindingFilters& AddProductName(ProductNameT&& value) { m_productNameHasBeenSet = true; m_productName.emplace_back(std::forward<ProductNameT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The name of the findings provider (company) that owns the solution (product)
     * that generates findings.</p>
     */
    inline const Aws::Vector<StringFilter>& GetCompanyName() const { return m_companyName; }
    inline bool CompanyNameHasBeenSet() const { return m_companyNameHasBeenSet; }
    template<typename CompanyNameT = Aws::Vector<StringFilter>>
    void SetCompanyName(CompanyNameT&& value) { m_companyNameHasBeenSet = true; m_companyName = std::forward<CompanyNameT>(value); }
    template<typename CompanyNameT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithCompanyName(CompanyNameT&& value) { SetCompanyName(std::forward<CompanyNameT>(value)); return *this;}
    template<typename CompanyNameT = StringFilter>
    AwsSecurityFindingFilters& AddCompanyName(CompanyNameT&& value) { m_companyNameHasBeenSet = true; m_companyName.emplace_back(std::forward<CompanyNameT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>A list of name/value string pairs associated with the finding. These are
     * custom, user-defined fields added to a finding. </p>
     */
    inline const Aws::Vector<MapFilter>& GetUserDefinedFields() const { return m_userDefinedFields; }
    inline bool UserDefinedFieldsHasBeenSet() const { return m_userDefinedFieldsHasBeenSet; }
    template<typename UserDefinedFieldsT = Aws::Vector<MapFilter>>
    void SetUserDefinedFields(UserDefinedFieldsT&& value) { m_userDefinedFieldsHasBeenSet = true; m_userDefinedFields = std::forward<UserDefinedFieldsT>(value); }
    template<typename UserDefinedFieldsT = Aws::Vector<MapFilter>>
    AwsSecurityFindingFilters& WithUserDefinedFields(UserDefinedFieldsT&& value) { SetUserDefinedFields(std::forward<UserDefinedFieldsT>(value)); return *this;}
    template<typename UserDefinedFieldsT = MapFilter>
    AwsSecurityFindingFilters& AddUserDefinedFields(UserDefinedFieldsT&& value) { m_userDefinedFieldsHasBeenSet = true; m_userDefinedFields.emplace_back(std::forward<UserDefinedFieldsT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The name of the malware that was observed.</p>
     */
    inline const Aws::Vector<StringFilter>& GetMalwareName() const { return m_malwareName; }
    inline bool MalwareNameHasBeenSet() const { return m_malwareNameHasBeenSet; }
    template<typename MalwareNameT = Aws::Vector<StringFilter>>
    void SetMalwareName(MalwareNameT&& value) { m_malwareNameHasBeenSet = true; m_malwareName = std::forward<MalwareNameT>(value); }
    template<typename MalwareNameT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithMalwareName(MalwareNameT&& value) { SetMalwareName(std::forward<MalwareNameT>(value)); return *this;}
    template<typename MalwareNameT = StringFilter>
    AwsSecurityFindingFilters& AddMalwareName(MalwareNameT&& value) { m_malwareNameHasBeenSet = true; m_malwareName.emplace_back(std::forward<MalwareNameT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The type of the malware that was observed.</p>
     */
    inline const Aws::Vector<StringFilter>& GetMalwareType() const { return m_malwareType; }
    inline bool MalwareTypeHasBeenSet() const { return m_malwareTypeHasBeenSet; }
    template<typename MalwareTypeT = Aws::Vector<StringFilter>>
    void SetMalwareType(MalwareTypeT&& value) { m_malwareTypeHasBeenSet = true; m_malwareType = std::forward<MalwareTypeT>(value); }
    template<typename MalwareTypeT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithMalwareType(MalwareTypeT&& value) { SetMalwareType(std::forward<MalwareTypeT>(value)); return *this;}
    template<typename MalwareTypeT = StringFilter>
    AwsSecurityFindingFilters& AddMalwareType(MalwareTypeT&& value) { m_malwareTypeHasBeenSet = true; m_malwareType.emplace_back(std::forward<MalwareTypeT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The filesystem path of the malware that was observed.</p>
     */
    inline const Aws::Vector<StringFilter>& GetMalwarePath() const { return m_malwarePath; }
    inline bool MalwarePathHasBeenSet() const { return m_malwarePathHasBeenSet; }
    template<typename MalwarePathT = Aws::Vector<StringFilter>>
    void SetMalwarePath(MalwarePathT&& value) { m_malwarePathHasBeenSet = true; m_malwarePath = std::forward<MalwarePathT>(value); }
    template<typename MalwarePathT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithMalwarePath(MalwarePathT&& value) { SetMalwarePath(std::forward<MalwarePathT>(value)); return *this;}
    template<typename MalwarePathT = StringFilter>
    AwsSecurityFindingFilters& AddMalwarePath(MalwarePathT&& value) { m_malwarePathHasBeenSet = true; m_malwarePath.emplace_back(std::forward<MalwarePathT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The state of the malware that was observed.</p>
     */
    inline const Aws::Vector<StringFilter>& GetMalwareState() const { return m_malwareState; }
    inline bool MalwareStateHasBeenSet() const { return m_malwareStateHasBeenSet; }
    template<typename MalwareStateT = Aws::Vector<StringFilter>>
    void SetMalwareState(MalwareStateT&& value) { m_malwareStateHasBeenSet = true; m_malwareState = std::forward<MalwareStateT>(value); }
    template<typename MalwareStateT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithMalwareState(MalwareStateT&& value) { SetMalwareState(std::forward<MalwareStateT>(value)); return *this;}
    template<typename MalwareStateT = StringFilter>
    AwsSecurityFindingFilters& AddMalwareState(MalwareStateT&& value) { m_malwareStateHasBeenSet = true; m_malwareState.emplace_back(std::forward<MalwareStateT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>Indicates the direction of network traffic associated with a finding.</p>
     */
    inline const Aws::Vector<StringFilter>& GetNetworkDirection() const { return m_networkDirection; }
    inline bool NetworkDirectionHasBeenSet() const { return m_networkDirectionHasBeenSet; }
    template<typename NetworkDirectionT = Aws::Vector<StringFilter>>
    void SetNetworkDirection(NetworkDirectionT&& value) { m_networkDirectionHasBeenSet = true; m_networkDirection = std::forward<NetworkDirectionT>(value); }
    template<typename NetworkDirectionT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithNetworkDirection(NetworkDirectionT&& value) { SetNetworkDirection(std::forward<NetworkDirectionT>(value)); return *this;}
    template<typename NetworkDirectionT = StringFilter>
    AwsSecurityFindingFilters& AddNetworkDirection(NetworkDirectionT&& value) { m_networkDirectionHasBeenSet = true; m_networkDirection.emplace_back(std::forward<NetworkDirectionT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The protocol of network-related information about a finding.</p>
     */
    inline const Aws::Vector<StringFilter>& GetNetworkProtocol() const { return m_networkProtocol; }
    inline bool NetworkProtocolHasBeenSet() const { return m_networkProtocolHasBeenSet; }
    template<typename NetworkProtocolT = Aws::Vector<StringFilter>>
    void SetNetworkProtocol(NetworkProtocolT&& value) { m_networkProtocolHasBeenSet = true; m_networkProtocol = std::forward<NetworkProtocolT>(value); }
    template<typename NetworkProtocolT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithNetworkProtocol(NetworkProtocolT&& value) { SetNetworkProtocol(std::forward<NetworkProtocolT>(value)); return *this;}
    template<typename NetworkProtocolT = StringFilter>
    AwsSecurityFindingFilters& AddNetworkProtocol(NetworkProtocolT&& value) { m_networkProtocolHasBeenSet = true; m_networkProtocol.emplace_back(std::forward<NetworkProtocolT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The source IPv4 address of network-related information about a finding.</p>
     */
    inline const Aws::Vector<IpFilter>& GetNetworkSourceIpV4() const { return m_networkSourceIpV4; }
    inline bool NetworkSourceIpV4HasBeenSet() const { return m_networkSourceIpV4HasBeenSet; }
    template<typename NetworkSourceIpV4T = Aws::Vector<IpFilter>>
    void SetNetworkSourceIpV4(NetworkSourceIpV4T&& value) { m_networkSourceIpV4HasBeenSet = true; m_networkSourceIpV4 = std::forward<NetworkSourceIpV4T>(value); }
    template<typename NetworkSourceIpV4T = Aws::Vector<IpFilter>>
    AwsSecurityFindingFilters& WithNetworkSourceIpV4(NetworkSourceIpV4T&& value) { SetNetworkSourceIpV4(std::forward<NetworkSourceIpV4T>(value)); return *this;}
    template<typename NetworkSourceIpV4T = IpFilter>
    AwsSecurityFindingFilters& AddNetworkSourceIpV4(NetworkSourceIpV4T&& value) { m_networkSourceIpV4HasBeenSet = true; m_networkSourceIpV4.emplace_back(std::forward<NetworkSourceIpV4T>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The source IPv6 address of network-related information about a finding.</p>
     */
    inline const Aws::Vector<IpFilter>& GetNetworkSourceIpV6() const { return m_networkSourceIpV6; }
    inline bool NetworkSourceIpV6HasBeenSet() const { return m_networkSourceIpV6HasBeenSet; }
    template<typename NetworkSourceIpV6T = Aws::Vector<IpFilter>>
    void SetNetworkSourceIpV6(NetworkSourceIpV6T&& value) { m_networkSourceIpV6HasBeenSet = true; m_networkSourceIpV6 = std::forward<NetworkSourceIpV6T>(value); }
    template<typename NetworkSourceIpV6T = Aws::Vector<IpFilter>>
    AwsSecurityFindingFilters& WithNetworkSourceIpV6(NetworkSourceIpV6T&& value) { SetNetworkSourceIpV6(std::forward<NetworkSourceIpV6T>(value)); return *this;}
    template<typename NetworkSourceIpV6T = IpFilter>
    AwsSecurityFindingFilters& AddNetworkSourceIpV6(NetworkSourceIpV6T&& value) { m_networkSourceIpV6HasBeenSet = true; m_networkSourceIpV6.emplace_back(std::forward<NetworkSourceIpV6T>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The source port of network-related information about a finding.</p>
     */
    inline const Aws::Vector<NumberFilter>& GetNetworkSourcePort() const { return m_networkSourcePort; }
    inline bool NetworkSourcePortHasBeenSet() const { return m_networkSourcePortHasBeenSet; }
    template<typename NetworkSourcePortT = Aws::Vector<NumberFilter>>
    void SetNetworkSourcePort(NetworkSourcePortT&& value) { m_networkSourcePortHasBeenSet = true; m_networkSourcePort = std::forward<NetworkSourcePortT>(value); }
    template<typename NetworkSourcePortT = Aws::Vector<NumberFilter>>
    AwsSecurityFindingFilters& WithNetworkSourcePort(NetworkSourcePortT&& value) { SetNetworkSourcePort(std::forward<NetworkSourcePortT>(value)); return *this;}
    template<typename NetworkSourcePortT = NumberFilter>
    AwsSecurityFindingFilters& AddNetworkSourcePort(NetworkSourcePortT&& value) { m_networkSourcePortHasBeenSet = true; m_networkSourcePort.emplace_back(std::forward<NetworkSourcePortT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The source domain of network-related information about a finding.</p>
     */
    inline const Aws::Vector<StringFilter>& GetNetworkSourceDomain() const { return m_networkSourceDomain; }
    inline bool NetworkSourceDomainHasBeenSet() const { return m_networkSourceDomainHasBeenSet; }
    template<typename NetworkSourceDomainT = Aws::Vector<StringFilter>>
    void SetNetworkSourceDomain(NetworkSourceDomainT&& value) { m_networkSourceDomainHasBeenSet = true; m_networkSourceDomain = std::forward<NetworkSourceDomainT>(value); }
    template<typename NetworkSourceDomainT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithNetworkSourceDomain(NetworkSourceDomainT&& value) { SetNetworkSourceDomain(std::forward<NetworkSourceDomainT>(value)); return *this;}
    template<typename NetworkSourceDomainT = StringFilter>
    AwsSecurityFindingFilters& AddNetworkSourceDomain(NetworkSourceDomainT&& value) { m_networkSourceDomainHasBeenSet = true; m_networkSourceDomain.emplace_back(std::forward<NetworkSourceDomainT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The source media access control (MAC) address of network-related information
     * about a finding.</p>
     */
    inline const Aws::Vector<StringFilter>& GetNetworkSourceMac() const { return m_networkSourceMac; }
    inline bool NetworkSourceMacHasBeenSet() const { return m_networkSourceMacHasBeenSet; }
    template<typename NetworkSourceMacT = Aws::Vector<StringFilter>>
    void SetNetworkSourceMac(NetworkSourceMacT&& value) { m_networkSourceMacHasBeenSet = true; m_networkSourceMac = std::forward<NetworkSourceMacT>(value); }
    template<typename NetworkSourceMacT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithNetworkSourceMac(NetworkSourceMacT&& value) { SetNetworkSourceMac(std::forward<NetworkSourceMacT>(value)); return *this;}
    template<typename NetworkSourceMacT = StringFilter>
    AwsSecurityFindingFilters& AddNetworkSourceMac(NetworkSourceMacT&& value) { m_networkSourceMacHasBeenSet = true; m_networkSourceMac.emplace_back(std::forward<NetworkSourceMacT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The destination IPv4 address of network-related information about a
     * finding.</p>
     */
    inline const Aws::Vector<IpFilter>& GetNetworkDestinationIpV4() const { return m_networkDestinationIpV4; }
    inline bool NetworkDestinationIpV4HasBeenSet() const { return m_networkDestinationIpV4HasBeenSet; }
    template<typename NetworkDestinationIpV4T = Aws::Vector<IpFilter>>
    void SetNetworkDestinationIpV4(NetworkDestinationIpV4T&& value) { m_networkDestinationIpV4HasBeenSet = true; m_networkDestinationIpV4 = std::forward<NetworkDestinationIpV4T>(value); }
    template<typename NetworkDestinationIpV4T = Aws::Vector<IpFilter>>
    AwsSecurityFindingFilters& WithNetworkDestinationIpV4(NetworkDestinationIpV4T&& value) { SetNetworkDestinationIpV4(std::forward<NetworkDestinationIpV4T>(value)); return *this;}
    template<typename NetworkDestinationIpV4T = IpFilter>
    AwsSecurityFindingFilters& AddNetworkDestinationIpV4(NetworkDestinationIpV4T&& value) { m_networkDestinationIpV4HasBeenSet = true; m_networkDestinationIpV4.emplace_back(std::forward<NetworkDestinationIpV4T>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The destination IPv6 address of network-related information about a
     * finding.</p>
     */
    inline const Aws::Vector<IpFilter>& GetNetworkDestinationIpV6() const { return m_networkDestinationIpV6; }
    inline bool NetworkDestinationIpV6HasBeenSet() const { return m_networkDestinationIpV6HasBeenSet; }
    template<typename NetworkDestinationIpV6T = Aws::Vector<IpFilter>>
    void SetNetworkDestinationIpV6(NetworkDestinationIpV6T&& value) { m_networkDestinationIpV6HasBeenSet = true; m_networkDestinationIpV6 = std::forward<NetworkDestinationIpV6T>(value); }
    template<typename NetworkDestinationIpV6T = Aws::Vector<IpFilter>>
    AwsSecurityFindingFilters& WithNetworkDestinationIpV6(NetworkDestinationIpV6T&& value) { SetNetworkDestinationIpV6(std::forward<NetworkDestinationIpV6T>(value)); return *this;}
    template<typename NetworkDestinationIpV6T = IpFilter>
    AwsSecurityFindingFilters& AddNetworkDestinationIpV6(NetworkDestinationIpV6T&& value) { m_networkDestinationIpV6HasBeenSet = true; m_networkDestinationIpV6.emplace_back(std::forward<NetworkDestinationIpV6T>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The destination port of network-related information about a finding.</p>
     */
    inline const Aws::Vector<NumberFilter>& GetNetworkDestinationPort() const { return m_networkDestinationPort; }
    inline bool NetworkDestinationPortHasBeenSet() const { return m_networkDestinationPortHasBeenSet; }
    template<typename NetworkDestinationPortT = Aws::Vector<NumberFilter>>
    void SetNetworkDestinationPort(NetworkDestinationPortT&& value) { m_networkDestinationPortHasBeenSet = true; m_networkDestinationPort = std::forward<NetworkDestinationPortT>(value); }
    template<typename NetworkDestinationPortT = Aws::Vector<NumberFilter>>
    AwsSecurityFindingFilters& WithNetworkDestinationPort(NetworkDestinationPortT&& value) { SetNetworkDestinationPort(std::forward<NetworkDestinationPortT>(value)); return *this;}
    template<typename NetworkDestinationPortT = NumberFilter>
    AwsSecurityFindingFilters& AddNetworkDestinationPort(NetworkDestinationPortT&& value) { m_networkDestinationPortHasBeenSet = true; m_networkDestinationPort.emplace_back(std::forward<NetworkDestinationPortT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The destination domain of network-related information about a finding.</p>
     */
    inline const Aws::Vector<StringFilter>& GetNetworkDestinationDomain() const { return m_networkDestinationDomain; }
    inline bool NetworkDestinationDomainHasBeenSet() const { return m_networkDestinationDomainHasBeenSet; }
    template<typename NetworkDestinationDomainT = Aws::Vector<StringFilter>>
    void SetNetworkDestinationDomain(NetworkDestinationDomainT&& value) { m_networkDestinationDomainHasBeenSet = true; m_networkDestinationDomain = std::forward<NetworkDestinationDomainT>(value); }
    template<typename NetworkDestinationDomainT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithNetworkDestinationDomain(NetworkDestinationDomainT&& value) { SetNetworkDestinationDomain(std::forward<NetworkDestinationDomainT>(value)); return *this;}
    template<typename NetworkDestinationDomainT = StringFilter>
    AwsSecurityFindingFilters& AddNetworkDestinationDomain(NetworkDestinationDomainT&& value) { m_networkDestinationDomainHasBeenSet = true; m_networkDestinationDomain.emplace_back(std::forward<NetworkDestinationDomainT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The name of the process.</p>
     */
    inline const Aws::Vector<StringFilter>& GetProcessName() const { return m_processName; }
    inline bool ProcessNameHasBeenSet() const { return m_processNameHasBeenSet; }
    template<typename ProcessNameT = Aws::Vector<StringFilter>>
    void SetProcessName(ProcessNameT&& value) { m_processNameHasBeenSet = true; m_processName = std::forward<ProcessNameT>(value); }
    template<typename ProcessNameT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithProcessName(ProcessNameT&& value) { SetProcessName(std::forward<ProcessNameT>(value)); return *this;}
    template<typename ProcessNameT = StringFilter>
    AwsSecurityFindingFilters& AddProcessName(ProcessNameT&& value) { m_processNameHasBeenSet = true; m_processName.emplace_back(std::forward<ProcessNameT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The path to the process executable.</p>
     */
    inline const Aws::Vector<StringFilter>& GetProcessPath() const { return m_processPath; }
    inline bool ProcessPathHasBeenSet() const { return m_processPathHasBeenSet; }
    template<typename ProcessPathT = Aws::Vector<StringFilter>>
    void SetProcessPath(ProcessPathT&& value) { m_processPathHasBeenSet = true; m_processPath = std::forward<ProcessPathT>(value); }
    template<typename ProcessPathT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithProcessPath(ProcessPathT&& value) { SetProcessPath(std::forward<ProcessPathT>(value)); return *this;}
    template<typename ProcessPathT = StringFilter>
    AwsSecurityFindingFilters& AddProcessPath(ProcessPathT&& value) { m_processPathHasBeenSet = true; m_processPath.emplace_back(std::forward<ProcessPathT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The process ID.</p>
     */
    inline const Aws::Vector<NumberFilter>& GetProcessPid() const { return m_processPid; }
    inline bool ProcessPidHasBeenSet() const { return m_processPidHasBeenSet; }
    template<typename ProcessPidT = Aws::Vector<NumberFilter>>
    void SetProcessPid(ProcessPidT&& value) { m_processPidHasBeenSet = true; m_processPid = std::forward<ProcessPidT>(value); }
    template<typename ProcessPidT = Aws::Vector<NumberFilter>>
    AwsSecurityFindingFilters& WithProcessPid(ProcessPidT&& value) { SetProcessPid(std::forward<ProcessPidT>(value)); return *this;}
    template<typename ProcessPidT = NumberFilter>
    AwsSecurityFindingFilters& AddProcessPid(ProcessPidT&& value) { m_processPidHasBeenSet = true; m_processPid.emplace_back(std::forward<ProcessPidT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The parent process ID. This field accepts positive integers between
     * <code>O</code> and <code>2147483647</code>.</p>
     */
    inline const Aws::Vector<NumberFilter>& GetProcessParentPid() const { return m_processParentPid; }
    inline bool ProcessParentPidHasBeenSet() const { return m_processParentPidHasBeenSet; }
    template<typename ProcessParentPidT = Aws::Vector<NumberFilter>>
    void SetProcessParentPid(ProcessParentPidT&& value) { m_processParentPidHasBeenSet = true; m_processParentPid = std::forward<ProcessParentPidT>(value); }
    template<typename ProcessParentPidT = Aws::Vector<NumberFilter>>
    AwsSecurityFindingFilters& WithProcessParentPid(ProcessParentPidT&& value) { SetProcessParentPid(std::forward<ProcessParentPidT>(value)); return *this;}
    template<typename ProcessParentPidT = NumberFilter>
    AwsSecurityFindingFilters& AddProcessParentPid(ProcessParentPidT&& value) { m_processParentPidHasBeenSet = true; m_processParentPid.emplace_back(std::forward<ProcessParentPidT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>A timestamp that identifies when the process was launched.</p> <p>For more
     * information about the validation and formatting of timestamp fields in Security
     * Hub, see <a
     * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/Welcome.html#timestamps">Timestamps</a>.</p>
     */
    inline const Aws::Vector<DateFilter>& GetProcessLaunchedAt() const { return m_processLaunchedAt; }
    inline bool ProcessLaunchedAtHasBeenSet() const { return m_processLaunchedAtHasBeenSet; }
    template<typename ProcessLaunchedAtT = Aws::Vector<DateFilter>>
    void SetProcessLaunchedAt(ProcessLaunchedAtT&& value) { m_processLaunchedAtHasBeenSet = true; m_processLaunchedAt = std::forward<ProcessLaunchedAtT>(value); }
    template<typename ProcessLaunchedAtT = Aws::Vector<DateFilter>>
    AwsSecurityFindingFilters& WithProcessLaunchedAt(ProcessLaunchedAtT&& value) { SetProcessLaunchedAt(std::forward<ProcessLaunchedAtT>(value)); return *this;}
    template<typename ProcessLaunchedAtT = DateFilter>
    AwsSecurityFindingFilters& AddProcessLaunchedAt(ProcessLaunchedAtT&& value) { m_processLaunchedAtHasBeenSet = true; m_processLaunchedAt.emplace_back(std::forward<ProcessLaunchedAtT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>A timestamp that identifies when the process was terminated.</p> <p>For more
     * information about the validation and formatting of timestamp fields in Security
     * Hub, see <a
     * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/Welcome.html#timestamps">Timestamps</a>.</p>
     */
    inline const Aws::Vector<DateFilter>& GetProcessTerminatedAt() const { return m_processTerminatedAt; }
    inline bool ProcessTerminatedAtHasBeenSet() const { return m_processTerminatedAtHasBeenSet; }
    template<typename ProcessTerminatedAtT = Aws::Vector<DateFilter>>
    void SetProcessTerminatedAt(ProcessTerminatedAtT&& value) { m_processTerminatedAtHasBeenSet = true; m_processTerminatedAt = std::forward<ProcessTerminatedAtT>(value); }
    template<typename ProcessTerminatedAtT = Aws::Vector<DateFilter>>
    AwsSecurityFindingFilters& WithProcessTerminatedAt(ProcessTerminatedAtT&& value) { SetProcessTerminatedAt(std::forward<ProcessTerminatedAtT>(value)); return *this;}
    template<typename ProcessTerminatedAtT = DateFilter>
    AwsSecurityFindingFilters& AddProcessTerminatedAt(ProcessTerminatedAtT&& value) { m_processTerminatedAtHasBeenSet = true; m_processTerminatedAt.emplace_back(std::forward<ProcessTerminatedAtT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The type of a threat intelligence indicator.</p>
     */
    inline const Aws::Vector<StringFilter>& GetThreatIntelIndicatorType() const { return m_threatIntelIndicatorType; }
    inline bool ThreatIntelIndicatorTypeHasBeenSet() const { return m_threatIntelIndicatorTypeHasBeenSet; }
    template<typename ThreatIntelIndicatorTypeT = Aws::Vector<StringFilter>>
    void SetThreatIntelIndicatorType(ThreatIntelIndicatorTypeT&& value) { m_threatIntelIndicatorTypeHasBeenSet = true; m_threatIntelIndicatorType = std::forward<ThreatIntelIndicatorTypeT>(value); }
    template<typename ThreatIntelIndicatorTypeT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithThreatIntelIndicatorType(ThreatIntelIndicatorTypeT&& value) { SetThreatIntelIndicatorType(std::forward<ThreatIntelIndicatorTypeT>(value)); return *this;}
    template<typename ThreatIntelIndicatorTypeT = StringFilter>
    AwsSecurityFindingFilters& AddThreatIntelIndicatorType(ThreatIntelIndicatorTypeT&& value) { m_threatIntelIndicatorTypeHasBeenSet = true; m_threatIntelIndicatorType.emplace_back(std::forward<ThreatIntelIndicatorTypeT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The value of a threat intelligence indicator.</p>
     */
    inline const Aws::Vector<StringFilter>& GetThreatIntelIndicatorValue() const { return m_threatIntelIndicatorValue; }
    inline bool ThreatIntelIndicatorValueHasBeenSet() const { return m_threatIntelIndicatorValueHasBeenSet; }
    template<typename ThreatIntelIndicatorValueT = Aws::Vector<StringFilter>>
    void SetThreatIntelIndicatorValue(ThreatIntelIndicatorValueT&& value) { m_threatIntelIndicatorValueHasBeenSet = true; m_threatIntelIndicatorValue = std::forward<ThreatIntelIndicatorValueT>(value); }
    template<typename ThreatIntelIndicatorValueT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithThreatIntelIndicatorValue(ThreatIntelIndicatorValueT&& value) { SetThreatIntelIndicatorValue(std::forward<ThreatIntelIndicatorValueT>(value)); return *this;}
    template<typename ThreatIntelIndicatorValueT = StringFilter>
    AwsSecurityFindingFilters& AddThreatIntelIndicatorValue(ThreatIntelIndicatorValueT&& value) { m_threatIntelIndicatorValueHasBeenSet = true; m_threatIntelIndicatorValue.emplace_back(std::forward<ThreatIntelIndicatorValueT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The category of a threat intelligence indicator.</p>
     */
    inline const Aws::Vector<StringFilter>& GetThreatIntelIndicatorCategory() const { return m_threatIntelIndicatorCategory; }
    inline bool ThreatIntelIndicatorCategoryHasBeenSet() const { return m_threatIntelIndicatorCategoryHasBeenSet; }
    template<typename ThreatIntelIndicatorCategoryT = Aws::Vector<StringFilter>>
    void SetThreatIntelIndicatorCategory(ThreatIntelIndicatorCategoryT&& value) { m_threatIntelIndicatorCategoryHasBeenSet = true; m_threatIntelIndicatorCategory = std::forward<ThreatIntelIndicatorCategoryT>(value); }
    template<typename ThreatIntelIndicatorCategoryT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithThreatIntelIndicatorCategory(ThreatIntelIndicatorCategoryT&& value) { SetThreatIntelIndicatorCategory(std::forward<ThreatIntelIndicatorCategoryT>(value)); return *this;}
    template<typename ThreatIntelIndicatorCategoryT = StringFilter>
    AwsSecurityFindingFilters& AddThreatIntelIndicatorCategory(ThreatIntelIndicatorCategoryT&& value) { m_threatIntelIndicatorCategoryHasBeenSet = true; m_threatIntelIndicatorCategory.emplace_back(std::forward<ThreatIntelIndicatorCategoryT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>A timestamp that identifies the last observation of a threat intelligence
     * indicator.</p> <p>For more information about the validation and formatting of
     * timestamp fields in Security Hub, see <a
     * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/Welcome.html#timestamps">Timestamps</a>.</p>
     */
    inline const Aws::Vector<DateFilter>& GetThreatIntelIndicatorLastObservedAt() const { return m_threatIntelIndicatorLastObservedAt; }
    inline bool ThreatIntelIndicatorLastObservedAtHasBeenSet() const { return m_threatIntelIndicatorLastObservedAtHasBeenSet; }
    template<typename ThreatIntelIndicatorLastObservedAtT = Aws::Vector<DateFilter>>
    void SetThreatIntelIndicatorLastObservedAt(ThreatIntelIndicatorLastObservedAtT&& value) { m_threatIntelIndicatorLastObservedAtHasBeenSet = true; m_threatIntelIndicatorLastObservedAt = std::forward<ThreatIntelIndicatorLastObservedAtT>(value); }
    template<typename ThreatIntelIndicatorLastObservedAtT = Aws::Vector<DateFilter>>
    AwsSecurityFindingFilters& WithThreatIntelIndicatorLastObservedAt(ThreatIntelIndicatorLastObservedAtT&& value) { SetThreatIntelIndicatorLastObservedAt(std::forward<ThreatIntelIndicatorLastObservedAtT>(value)); return *this;}
    template<typename ThreatIntelIndicatorLastObservedAtT = DateFilter>
    AwsSecurityFindingFilters& AddThreatIntelIndicatorLastObservedAt(ThreatIntelIndicatorLastObservedAtT&& value) { m_threatIntelIndicatorLastObservedAtHasBeenSet = true; m_threatIntelIndicatorLastObservedAt.emplace_back(std::forward<ThreatIntelIndicatorLastObservedAtT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The source of the threat intelligence.</p>
     */
    inline const Aws::Vector<StringFilter>& GetThreatIntelIndicatorSource() const { return m_threatIntelIndicatorSource; }
    inline bool ThreatIntelIndicatorSourceHasBeenSet() const { return m_threatIntelIndicatorSourceHasBeenSet; }
    template<typename ThreatIntelIndicatorSourceT = Aws::Vector<StringFilter>>
    void SetThreatIntelIndicatorSource(ThreatIntelIndicatorSourceT&& value) { m_threatIntelIndicatorSourceHasBeenSet = true; m_threatIntelIndicatorSource = std::forward<ThreatIntelIndicatorSourceT>(value); }
    template<typename ThreatIntelIndicatorSourceT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithThreatIntelIndicatorSource(ThreatIntelIndicatorSourceT&& value) { SetThreatIntelIndicatorSource(std::forward<ThreatIntelIndicatorSourceT>(value)); return *this;}
    template<typename ThreatIntelIndicatorSourceT = StringFilter>
    AwsSecurityFindingFilters& AddThreatIntelIndicatorSource(ThreatIntelIndicatorSourceT&& value) { m_threatIntelIndicatorSourceHasBeenSet = true; m_threatIntelIndicatorSource.emplace_back(std::forward<ThreatIntelIndicatorSourceT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The URL for more details from the source of the threat intelligence.</p>
     */
    inline const Aws::Vector<StringFilter>& GetThreatIntelIndicatorSourceUrl() const { return m_threatIntelIndicatorSourceUrl; }
    inline bool ThreatIntelIndicatorSourceUrlHasBeenSet() const { return m_threatIntelIndicatorSourceUrlHasBeenSet; }
    template<typename ThreatIntelIndicatorSourceUrlT = Aws::Vector<StringFilter>>
    void SetThreatIntelIndicatorSourceUrl(ThreatIntelIndicatorSourceUrlT&& value) { m_threatIntelIndicatorSourceUrlHasBeenSet = true; m_threatIntelIndicatorSourceUrl = std::forward<ThreatIntelIndicatorSourceUrlT>(value); }
    template<typename ThreatIntelIndicatorSourceUrlT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithThreatIntelIndicatorSourceUrl(ThreatIntelIndicatorSourceUrlT&& value) { SetThreatIntelIndicatorSourceUrl(std::forward<ThreatIntelIndicatorSourceUrlT>(value)); return *this;}
    template<typename ThreatIntelIndicatorSourceUrlT = StringFilter>
    AwsSecurityFindingFilters& AddThreatIntelIndicatorSourceUrl(ThreatIntelIndicatorSourceUrlT&& value) { m_threatIntelIndicatorSourceUrlHasBeenSet = true; m_threatIntelIndicatorSourceUrl.emplace_back(std::forward<ThreatIntelIndicatorSourceUrlT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>Specifies the type of the resource that details are provided for.</p>
     */
    inline const Aws::Vector<StringFilter>& GetResourceType() const { return m_resourceType; }
    inline bool ResourceTypeHasBeenSet() const { return m_resourceTypeHasBeenSet; }
    template<typename ResourceTypeT = Aws::Vector<StringFilter>>
    void SetResourceType(ResourceTypeT&& value) { m_resourceTypeHasBeenSet = true; m_resourceType = std::forward<ResourceTypeT>(value); }
    template<typename ResourceTypeT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithResourceType(ResourceTypeT&& value) { SetResourceType(std::forward<ResourceTypeT>(value)); return *this;}
    template<typename ResourceTypeT = StringFilter>
    AwsSecurityFindingFilters& AddResourceType(ResourceTypeT&& value) { m_resourceTypeHasBeenSet = true; m_resourceType.emplace_back(std::forward<ResourceTypeT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The canonical identifier for the given resource type.</p>
     */
    inline const Aws::Vector<StringFilter>& GetResourceId() const { return m_resourceId; }
    inline bool ResourceIdHasBeenSet() const { return m_resourceIdHasBeenSet; }
    template<typename ResourceIdT = Aws::Vector<StringFilter>>
    void SetResourceId(ResourceIdT&& value) { m_resourceIdHasBeenSet = true; m_resourceId = std::forward<ResourceIdT>(value); }
    template<typename ResourceIdT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithResourceId(ResourceIdT&& value) { SetResourceId(std::forward<ResourceIdT>(value)); return *this;}
    template<typename ResourceIdT = StringFilter>
    AwsSecurityFindingFilters& AddResourceId(ResourceIdT&& value) { m_resourceIdHasBeenSet = true; m_resourceId.emplace_back(std::forward<ResourceIdT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The canonical Amazon Web Services partition name that the Region is assigned
     * to.</p>
     */
    inline const Aws::Vector<StringFilter>& GetResourcePartition() const { return m_resourcePartition; }
    inline bool ResourcePartitionHasBeenSet() const { return m_resourcePartitionHasBeenSet; }
    template<typename ResourcePartitionT = Aws::Vector<StringFilter>>
    void SetResourcePartition(ResourcePartitionT&& value) { m_resourcePartitionHasBeenSet = true; m_resourcePartition = std::forward<ResourcePartitionT>(value); }
    template<typename ResourcePartitionT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithResourcePartition(ResourcePartitionT&& value) { SetResourcePartition(std::forward<ResourcePartitionT>(value)); return *this;}
    template<typename ResourcePartitionT = StringFilter>
    AwsSecurityFindingFilters& AddResourcePartition(ResourcePartitionT&& value) { m_resourcePartitionHasBeenSet = true; m_resourcePartition.emplace_back(std::forward<ResourcePartitionT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The canonical Amazon Web Services external Region name where this resource is
     * located.</p>
     */
    inline const Aws::Vector<StringFilter>& GetResourceRegion() const { return m_resourceRegion; }
    inline bool ResourceRegionHasBeenSet() const { return m_resourceRegionHasBeenSet; }
    template<typename ResourceRegionT = Aws::Vector<StringFilter>>
    void SetResourceRegion(ResourceRegionT&& value) { m_resourceRegionHasBeenSet = true; m_resourceRegion = std::forward<ResourceRegionT>(value); }
    template<typename ResourceRegionT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithResourceRegion(ResourceRegionT&& value) { SetResourceRegion(std::forward<ResourceRegionT>(value)); return *this;}
    template<typename ResourceRegionT = StringFilter>
    AwsSecurityFindingFilters& AddResourceRegion(ResourceRegionT&& value) { m_resourceRegionHasBeenSet = true; m_resourceRegion.emplace_back(std::forward<ResourceRegionT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>A list of Amazon Web Services tags associated with a resource at the time the
     * finding was processed.</p>
     */
    inline const Aws::Vector<MapFilter>& GetResourceTags() const { return m_resourceTags; }
    inline bool ResourceTagsHasBeenSet() const { return m_resourceTagsHasBeenSet; }
    template<typename ResourceTagsT = Aws::Vector<MapFilter>>
    void SetResourceTags(ResourceTagsT&& value) { m_resourceTagsHasBeenSet = true; m_resourceTags = std::forward<ResourceTagsT>(value); }
    template<typename ResourceTagsT = Aws::Vector<MapFilter>>
    AwsSecurityFindingFilters& WithResourceTags(ResourceTagsT&& value) { SetResourceTags(std::forward<ResourceTagsT>(value)); return *this;}
    template<typename ResourceTagsT = MapFilter>
    AwsSecurityFindingFilters& AddResourceTags(ResourceTagsT&& value) { m_resourceTagsHasBeenSet = true; m_resourceTags.emplace_back(std::forward<ResourceTagsT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The instance type of the instance.</p>
     */
    inline const Aws::Vector<StringFilter>& GetResourceAwsEc2InstanceType() const { return m_resourceAwsEc2InstanceType; }
    inline bool ResourceAwsEc2InstanceTypeHasBeenSet() const { return m_resourceAwsEc2InstanceTypeHasBeenSet; }
    template<typename ResourceAwsEc2InstanceTypeT = Aws::Vector<StringFilter>>
    void SetResourceAwsEc2InstanceType(ResourceAwsEc2InstanceTypeT&& value) { m_resourceAwsEc2InstanceTypeHasBeenSet = true; m_resourceAwsEc2InstanceType = std::forward<ResourceAwsEc2InstanceTypeT>(value); }
    template<typename ResourceAwsEc2InstanceTypeT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithResourceAwsEc2InstanceType(ResourceAwsEc2InstanceTypeT&& value) { SetResourceAwsEc2InstanceType(std::forward<ResourceAwsEc2InstanceTypeT>(value)); return *this;}
    template<typename ResourceAwsEc2InstanceTypeT = StringFilter>
    AwsSecurityFindingFilters& AddResourceAwsEc2InstanceType(ResourceAwsEc2InstanceTypeT&& value) { m_resourceAwsEc2InstanceTypeHasBeenSet = true; m_resourceAwsEc2InstanceType.emplace_back(std::forward<ResourceAwsEc2InstanceTypeT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The Amazon Machine Image (AMI) ID of the instance.</p>
     */
    inline const Aws::Vector<StringFilter>& GetResourceAwsEc2InstanceImageId() const { return m_resourceAwsEc2InstanceImageId; }
    inline bool ResourceAwsEc2InstanceImageIdHasBeenSet() const { return m_resourceAwsEc2InstanceImageIdHasBeenSet; }
    template<typename ResourceAwsEc2InstanceImageIdT = Aws::Vector<StringFilter>>
    void SetResourceAwsEc2InstanceImageId(ResourceAwsEc2InstanceImageIdT&& value) { m_resourceAwsEc2InstanceImageIdHasBeenSet = true; m_resourceAwsEc2InstanceImageId = std::forward<ResourceAwsEc2InstanceImageIdT>(value); }
    template<typename ResourceAwsEc2InstanceImageIdT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithResourceAwsEc2InstanceImageId(ResourceAwsEc2InstanceImageIdT&& value) { SetResourceAwsEc2InstanceImageId(std::forward<ResourceAwsEc2InstanceImageIdT>(value)); return *this;}
    template<typename ResourceAwsEc2InstanceImageIdT = StringFilter>
    AwsSecurityFindingFilters& AddResourceAwsEc2InstanceImageId(ResourceAwsEc2InstanceImageIdT&& value) { m_resourceAwsEc2InstanceImageIdHasBeenSet = true; m_resourceAwsEc2InstanceImageId.emplace_back(std::forward<ResourceAwsEc2InstanceImageIdT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The IPv4 addresses associated with the instance.</p>
     */
    inline const Aws::Vector<IpFilter>& GetResourceAwsEc2InstanceIpV4Addresses() const { return m_resourceAwsEc2InstanceIpV4Addresses; }
    inline bool ResourceAwsEc2InstanceIpV4AddressesHasBeenSet() const { return m_resourceAwsEc2InstanceIpV4AddressesHasBeenSet; }
    template<typename ResourceAwsEc2InstanceIpV4AddressesT = Aws::Vector<IpFilter>>
    void SetResourceAwsEc2InstanceIpV4Addresses(ResourceAwsEc2InstanceIpV4AddressesT&& value) { m_resourceAwsEc2InstanceIpV4AddressesHasBeenSet = true; m_resourceAwsEc2InstanceIpV4Addresses = std::forward<ResourceAwsEc2InstanceIpV4AddressesT>(value); }
    template<typename ResourceAwsEc2InstanceIpV4AddressesT = Aws::Vector<IpFilter>>
    AwsSecurityFindingFilters& WithResourceAwsEc2InstanceIpV4Addresses(ResourceAwsEc2InstanceIpV4AddressesT&& value) { SetResourceAwsEc2InstanceIpV4Addresses(std::forward<ResourceAwsEc2InstanceIpV4AddressesT>(value)); return *this;}
    template<typename ResourceAwsEc2InstanceIpV4AddressesT = IpFilter>
    AwsSecurityFindingFilters& AddResourceAwsEc2InstanceIpV4Addresses(ResourceAwsEc2InstanceIpV4AddressesT&& value) { m_resourceAwsEc2InstanceIpV4AddressesHasBeenSet = true; m_resourceAwsEc2InstanceIpV4Addresses.emplace_back(std::forward<ResourceAwsEc2InstanceIpV4AddressesT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The IPv6 addresses associated with the instance.</p>
     */
    inline const Aws::Vector<IpFilter>& GetResourceAwsEc2InstanceIpV6Addresses() const { return m_resourceAwsEc2InstanceIpV6Addresses; }
    inline bool ResourceAwsEc2InstanceIpV6AddressesHasBeenSet() const { return m_resourceAwsEc2InstanceIpV6AddressesHasBeenSet; }
    template<typename ResourceAwsEc2InstanceIpV6AddressesT = Aws::Vector<IpFilter>>
    void SetResourceAwsEc2InstanceIpV6Addresses(ResourceAwsEc2InstanceIpV6AddressesT&& value) { m_resourceAwsEc2InstanceIpV6AddressesHasBeenSet = true; m_resourceAwsEc2InstanceIpV6Addresses = std::forward<ResourceAwsEc2InstanceIpV6AddressesT>(value); }
    template<typename ResourceAwsEc2InstanceIpV6AddressesT = Aws::Vector<IpFilter>>
    AwsSecurityFindingFilters& WithResourceAwsEc2InstanceIpV6Addresses(ResourceAwsEc2InstanceIpV6AddressesT&& value) { SetResourceAwsEc2InstanceIpV6Addresses(std::forward<ResourceAwsEc2InstanceIpV6AddressesT>(value)); return *this;}
    template<typename ResourceAwsEc2InstanceIpV6AddressesT = IpFilter>
    AwsSecurityFindingFilters& AddResourceAwsEc2InstanceIpV6Addresses(ResourceAwsEc2InstanceIpV6AddressesT&& value) { m_resourceAwsEc2InstanceIpV6AddressesHasBeenSet = true; m_resourceAwsEc2InstanceIpV6Addresses.emplace_back(std::forward<ResourceAwsEc2InstanceIpV6AddressesT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The key name associated with the instance.</p>
     */
    inline const Aws::Vector<StringFilter>& GetResourceAwsEc2InstanceKeyName() const { return m_resourceAwsEc2InstanceKeyName; }
    inline bool ResourceAwsEc2InstanceKeyNameHasBeenSet() const { return m_resourceAwsEc2InstanceKeyNameHasBeenSet; }
    template<typename ResourceAwsEc2InstanceKeyNameT = Aws::Vector<StringFilter>>
    void SetResourceAwsEc2InstanceKeyName(ResourceAwsEc2InstanceKeyNameT&& value) { m_resourceAwsEc2InstanceKeyNameHasBeenSet = true; m_resourceAwsEc2InstanceKeyName = std::forward<ResourceAwsEc2InstanceKeyNameT>(value); }
    template<typename ResourceAwsEc2InstanceKeyNameT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithResourceAwsEc2InstanceKeyName(ResourceAwsEc2InstanceKeyNameT&& value) { SetResourceAwsEc2InstanceKeyName(std::forward<ResourceAwsEc2InstanceKeyNameT>(value)); return *this;}
    template<typename ResourceAwsEc2InstanceKeyNameT = StringFilter>
    AwsSecurityFindingFilters& AddResourceAwsEc2InstanceKeyName(ResourceAwsEc2InstanceKeyNameT&& value) { m_resourceAwsEc2InstanceKeyNameHasBeenSet = true; m_resourceAwsEc2InstanceKeyName.emplace_back(std::forward<ResourceAwsEc2InstanceKeyNameT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The IAM profile ARN of the instance.</p>
     */
    inline const Aws::Vector<StringFilter>& GetResourceAwsEc2InstanceIamInstanceProfileArn() const { return m_resourceAwsEc2InstanceIamInstanceProfileArn; }
    inline bool ResourceAwsEc2InstanceIamInstanceProfileArnHasBeenSet() const { return m_resourceAwsEc2InstanceIamInstanceProfileArnHasBeenSet; }
    template<typename ResourceAwsEc2InstanceIamInstanceProfileArnT = Aws::Vector<StringFilter>>
    void SetResourceAwsEc2InstanceIamInstanceProfileArn(ResourceAwsEc2InstanceIamInstanceProfileArnT&& value) { m_resourceAwsEc2InstanceIamInstanceProfileArnHasBeenSet = true; m_resourceAwsEc2InstanceIamInstanceProfileArn = std::forward<ResourceAwsEc2InstanceIamInstanceProfileArnT>(value); }
    template<typename ResourceAwsEc2InstanceIamInstanceProfileArnT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithResourceAwsEc2InstanceIamInstanceProfileArn(ResourceAwsEc2InstanceIamInstanceProfileArnT&& value) { SetResourceAwsEc2InstanceIamInstanceProfileArn(std::forward<ResourceAwsEc2InstanceIamInstanceProfileArnT>(value)); return *this;}
    template<typename ResourceAwsEc2InstanceIamInstanceProfileArnT = StringFilter>
    AwsSecurityFindingFilters& AddResourceAwsEc2InstanceIamInstanceProfileArn(ResourceAwsEc2InstanceIamInstanceProfileArnT&& value) { m_resourceAwsEc2InstanceIamInstanceProfileArnHasBeenSet = true; m_resourceAwsEc2InstanceIamInstanceProfileArn.emplace_back(std::forward<ResourceAwsEc2InstanceIamInstanceProfileArnT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The identifier of the VPC that the instance was launched in.</p>
     */
    inline const Aws::Vector<StringFilter>& GetResourceAwsEc2InstanceVpcId() const { return m_resourceAwsEc2InstanceVpcId; }
    inline bool ResourceAwsEc2InstanceVpcIdHasBeenSet() const { return m_resourceAwsEc2InstanceVpcIdHasBeenSet; }
    template<typename ResourceAwsEc2InstanceVpcIdT = Aws::Vector<StringFilter>>
    void SetResourceAwsEc2InstanceVpcId(ResourceAwsEc2InstanceVpcIdT&& value) { m_resourceAwsEc2InstanceVpcIdHasBeenSet = true; m_resourceAwsEc2InstanceVpcId = std::forward<ResourceAwsEc2InstanceVpcIdT>(value); }
    template<typename ResourceAwsEc2InstanceVpcIdT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithResourceAwsEc2InstanceVpcId(ResourceAwsEc2InstanceVpcIdT&& value) { SetResourceAwsEc2InstanceVpcId(std::forward<ResourceAwsEc2InstanceVpcIdT>(value)); return *this;}
    template<typename ResourceAwsEc2InstanceVpcIdT = StringFilter>
    AwsSecurityFindingFilters& AddResourceAwsEc2InstanceVpcId(ResourceAwsEc2InstanceVpcIdT&& value) { m_resourceAwsEc2InstanceVpcIdHasBeenSet = true; m_resourceAwsEc2InstanceVpcId.emplace_back(std::forward<ResourceAwsEc2InstanceVpcIdT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The identifier of the subnet that the instance was launched in.</p>
     */
    inline const Aws::Vector<StringFilter>& GetResourceAwsEc2InstanceSubnetId() const { return m_resourceAwsEc2InstanceSubnetId; }
    inline bool ResourceAwsEc2InstanceSubnetIdHasBeenSet() const { return m_resourceAwsEc2InstanceSubnetIdHasBeenSet; }
    template<typename ResourceAwsEc2InstanceSubnetIdT = Aws::Vector<StringFilter>>
    void SetResourceAwsEc2InstanceSubnetId(ResourceAwsEc2InstanceSubnetIdT&& value) { m_resourceAwsEc2InstanceSubnetIdHasBeenSet = true; m_resourceAwsEc2InstanceSubnetId = std::forward<ResourceAwsEc2InstanceSubnetIdT>(value); }
    template<typename ResourceAwsEc2InstanceSubnetIdT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithResourceAwsEc2InstanceSubnetId(ResourceAwsEc2InstanceSubnetIdT&& value) { SetResourceAwsEc2InstanceSubnetId(std::forward<ResourceAwsEc2InstanceSubnetIdT>(value)); return *this;}
    template<typename ResourceAwsEc2InstanceSubnetIdT = StringFilter>
    AwsSecurityFindingFilters& AddResourceAwsEc2InstanceSubnetId(ResourceAwsEc2InstanceSubnetIdT&& value) { m_resourceAwsEc2InstanceSubnetIdHasBeenSet = true; m_resourceAwsEc2InstanceSubnetId.emplace_back(std::forward<ResourceAwsEc2InstanceSubnetIdT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The date and time the instance was launched.</p>
     */
    inline const Aws::Vector<DateFilter>& GetResourceAwsEc2InstanceLaunchedAt() const { return m_resourceAwsEc2InstanceLaunchedAt; }
    inline bool ResourceAwsEc2InstanceLaunchedAtHasBeenSet() const { return m_resourceAwsEc2InstanceLaunchedAtHasBeenSet; }
    template<typename ResourceAwsEc2InstanceLaunchedAtT = Aws::Vector<DateFilter>>
    void SetResourceAwsEc2InstanceLaunchedAt(ResourceAwsEc2InstanceLaunchedAtT&& value) { m_resourceAwsEc2InstanceLaunchedAtHasBeenSet = true; m_resourceAwsEc2InstanceLaunchedAt = std::forward<ResourceAwsEc2InstanceLaunchedAtT>(value); }
    template<typename ResourceAwsEc2InstanceLaunchedAtT = Aws::Vector<DateFilter>>
    AwsSecurityFindingFilters& WithResourceAwsEc2InstanceLaunchedAt(ResourceAwsEc2InstanceLaunchedAtT&& value) { SetResourceAwsEc2InstanceLaunchedAt(std::forward<ResourceAwsEc2InstanceLaunchedAtT>(value)); return *this;}
    template<typename ResourceAwsEc2InstanceLaunchedAtT = DateFilter>
    AwsSecurityFindingFilters& AddResourceAwsEc2InstanceLaunchedAt(ResourceAwsEc2InstanceLaunchedAtT&& value) { m_resourceAwsEc2InstanceLaunchedAtHasBeenSet = true; m_resourceAwsEc2InstanceLaunchedAt.emplace_back(std::forward<ResourceAwsEc2InstanceLaunchedAtT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The canonical user ID of the owner of the S3 bucket.</p>
     */
    inline const Aws::Vector<StringFilter>& GetResourceAwsS3BucketOwnerId() const { return m_resourceAwsS3BucketOwnerId; }
    inline bool ResourceAwsS3BucketOwnerIdHasBeenSet() const { return m_resourceAwsS3BucketOwnerIdHasBeenSet; }
    template<typename ResourceAwsS3BucketOwnerIdT = Aws::Vector<StringFilter>>
    void SetResourceAwsS3BucketOwnerId(ResourceAwsS3BucketOwnerIdT&& value) { m_resourceAwsS3BucketOwnerIdHasBeenSet = true; m_resourceAwsS3BucketOwnerId = std::forward<ResourceAwsS3BucketOwnerIdT>(value); }
    template<typename ResourceAwsS3BucketOwnerIdT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithResourceAwsS3BucketOwnerId(ResourceAwsS3BucketOwnerIdT&& value) { SetResourceAwsS3BucketOwnerId(std::forward<ResourceAwsS3BucketOwnerIdT>(value)); return *this;}
    template<typename ResourceAwsS3BucketOwnerIdT = StringFilter>
    AwsSecurityFindingFilters& AddResourceAwsS3BucketOwnerId(ResourceAwsS3BucketOwnerIdT&& value) { m_resourceAwsS3BucketOwnerIdHasBeenSet = true; m_resourceAwsS3BucketOwnerId.emplace_back(std::forward<ResourceAwsS3BucketOwnerIdT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The display name of the owner of the S3 bucket.</p>
     */
    inline const Aws::Vector<StringFilter>& GetResourceAwsS3BucketOwnerName() const { return m_resourceAwsS3BucketOwnerName; }
    inline bool ResourceAwsS3BucketOwnerNameHasBeenSet() const { return m_resourceAwsS3BucketOwnerNameHasBeenSet; }
    template<typename ResourceAwsS3BucketOwnerNameT = Aws::Vector<StringFilter>>
    void SetResourceAwsS3BucketOwnerName(ResourceAwsS3BucketOwnerNameT&& value) { m_resourceAwsS3BucketOwnerNameHasBeenSet = true; m_resourceAwsS3BucketOwnerName = std::forward<ResourceAwsS3BucketOwnerNameT>(value); }
    template<typename ResourceAwsS3BucketOwnerNameT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithResourceAwsS3BucketOwnerName(ResourceAwsS3BucketOwnerNameT&& value) { SetResourceAwsS3BucketOwnerName(std::forward<ResourceAwsS3BucketOwnerNameT>(value)); return *this;}
    template<typename ResourceAwsS3BucketOwnerNameT = StringFilter>
    AwsSecurityFindingFilters& AddResourceAwsS3BucketOwnerName(ResourceAwsS3BucketOwnerNameT&& value) { m_resourceAwsS3BucketOwnerNameHasBeenSet = true; m_resourceAwsS3BucketOwnerName.emplace_back(std::forward<ResourceAwsS3BucketOwnerNameT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The name of the principal that is associated with an IAM access key.</p>
     */
    inline const Aws::Vector<StringFilter>& GetResourceAwsIamAccessKeyPrincipalName() const { return m_resourceAwsIamAccessKeyPrincipalName; }
    inline bool ResourceAwsIamAccessKeyPrincipalNameHasBeenSet() const { return m_resourceAwsIamAccessKeyPrincipalNameHasBeenSet; }
    template<typename ResourceAwsIamAccessKeyPrincipalNameT = Aws::Vector<StringFilter>>
    void SetResourceAwsIamAccessKeyPrincipalName(ResourceAwsIamAccessKeyPrincipalNameT&& value) { m_resourceAwsIamAccessKeyPrincipalNameHasBeenSet = true; m_resourceAwsIamAccessKeyPrincipalName = std::forward<ResourceAwsIamAccessKeyPrincipalNameT>(value); }
    template<typename ResourceAwsIamAccessKeyPrincipalNameT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithResourceAwsIamAccessKeyPrincipalName(ResourceAwsIamAccessKeyPrincipalNameT&& value) { SetResourceAwsIamAccessKeyPrincipalName(std::forward<ResourceAwsIamAccessKeyPrincipalNameT>(value)); return *this;}
    template<typename ResourceAwsIamAccessKeyPrincipalNameT = StringFilter>
    AwsSecurityFindingFilters& AddResourceAwsIamAccessKeyPrincipalName(ResourceAwsIamAccessKeyPrincipalNameT&& value) { m_resourceAwsIamAccessKeyPrincipalNameHasBeenSet = true; m_resourceAwsIamAccessKeyPrincipalName.emplace_back(std::forward<ResourceAwsIamAccessKeyPrincipalNameT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The status of the IAM access key related to a finding.</p>
     */
    inline const Aws::Vector<StringFilter>& GetResourceAwsIamAccessKeyStatus() const { return m_resourceAwsIamAccessKeyStatus; }
    inline bool ResourceAwsIamAccessKeyStatusHasBeenSet() const { return m_resourceAwsIamAccessKeyStatusHasBeenSet; }
    template<typename ResourceAwsIamAccessKeyStatusT = Aws::Vector<StringFilter>>
    void SetResourceAwsIamAccessKeyStatus(ResourceAwsIamAccessKeyStatusT&& value) { m_resourceAwsIamAccessKeyStatusHasBeenSet = true; m_resourceAwsIamAccessKeyStatus = std::forward<ResourceAwsIamAccessKeyStatusT>(value); }
    template<typename ResourceAwsIamAccessKeyStatusT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithResourceAwsIamAccessKeyStatus(ResourceAwsIamAccessKeyStatusT&& value) { SetResourceAwsIamAccessKeyStatus(std::forward<ResourceAwsIamAccessKeyStatusT>(value)); return *this;}
    template<typename ResourceAwsIamAccessKeyStatusT = StringFilter>
    AwsSecurityFindingFilters& AddResourceAwsIamAccessKeyStatus(ResourceAwsIamAccessKeyStatusT&& value) { m_resourceAwsIamAccessKeyStatusHasBeenSet = true; m_resourceAwsIamAccessKeyStatus.emplace_back(std::forward<ResourceAwsIamAccessKeyStatusT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The creation date/time of the IAM access key related to a finding.</p>
     */
    inline const Aws::Vector<DateFilter>& GetResourceAwsIamAccessKeyCreatedAt() const { return m_resourceAwsIamAccessKeyCreatedAt; }
    inline bool ResourceAwsIamAccessKeyCreatedAtHasBeenSet() const { return m_resourceAwsIamAccessKeyCreatedAtHasBeenSet; }
    template<typename ResourceAwsIamAccessKeyCreatedAtT = Aws::Vector<DateFilter>>
    void SetResourceAwsIamAccessKeyCreatedAt(ResourceAwsIamAccessKeyCreatedAtT&& value) { m_resourceAwsIamAccessKeyCreatedAtHasBeenSet = true; m_resourceAwsIamAccessKeyCreatedAt = std::forward<ResourceAwsIamAccessKeyCreatedAtT>(value); }
    template<typename ResourceAwsIamAccessKeyCreatedAtT = Aws::Vector<DateFilter>>
    AwsSecurityFindingFilters& WithResourceAwsIamAccessKeyCreatedAt(ResourceAwsIamAccessKeyCreatedAtT&& value) { SetResourceAwsIamAccessKeyCreatedAt(std::forward<ResourceAwsIamAccessKeyCreatedAtT>(value)); return *this;}
    template<typename ResourceAwsIamAccessKeyCreatedAtT = DateFilter>
    AwsSecurityFindingFilters& AddResourceAwsIamAccessKeyCreatedAt(ResourceAwsIamAccessKeyCreatedAtT&& value) { m_resourceAwsIamAccessKeyCreatedAtHasBeenSet = true; m_resourceAwsIamAccessKeyCreatedAt.emplace_back(std::forward<ResourceAwsIamAccessKeyCreatedAtT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The name of an IAM user.</p>
     */
    inline const Aws::Vector<StringFilter>& GetResourceAwsIamUserUserName() const { return m_resourceAwsIamUserUserName; }
    inline bool ResourceAwsIamUserUserNameHasBeenSet() const { return m_resourceAwsIamUserUserNameHasBeenSet; }
    template<typename ResourceAwsIamUserUserNameT = Aws::Vector<StringFilter>>
    void SetResourceAwsIamUserUserName(ResourceAwsIamUserUserNameT&& value) { m_resourceAwsIamUserUserNameHasBeenSet = true; m_resourceAwsIamUserUserName = std::forward<ResourceAwsIamUserUserNameT>(value); }
    template<typename ResourceAwsIamUserUserNameT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithResourceAwsIamUserUserName(ResourceAwsIamUserUserNameT&& value) { SetResourceAwsIamUserUserName(std::forward<ResourceAwsIamUserUserNameT>(value)); return *this;}
    template<typename ResourceAwsIamUserUserNameT = StringFilter>
    AwsSecurityFindingFilters& AddResourceAwsIamUserUserName(ResourceAwsIamUserUserNameT&& value) { m_resourceAwsIamUserUserNameHasBeenSet = true; m_resourceAwsIamUserUserName.emplace_back(std::forward<ResourceAwsIamUserUserNameT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The name of the container related to a finding.</p>
     */
    inline const Aws::Vector<StringFilter>& GetResourceContainerName() const { return m_resourceContainerName; }
    inline bool ResourceContainerNameHasBeenSet() const { return m_resourceContainerNameHasBeenSet; }
    template<typename ResourceContainerNameT = Aws::Vector<StringFilter>>
    void SetResourceContainerName(ResourceContainerNameT&& value) { m_resourceContainerNameHasBeenSet = true; m_resourceContainerName = std::forward<ResourceContainerNameT>(value); }
    template<typename ResourceContainerNameT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithResourceContainerName(ResourceContainerNameT&& value) { SetResourceContainerName(std::forward<ResourceContainerNameT>(value)); return *this;}
    template<typename ResourceContainerNameT = StringFilter>
    AwsSecurityFindingFilters& AddResourceContainerName(ResourceContainerNameT&& value) { m_resourceContainerNameHasBeenSet = true; m_resourceContainerName.emplace_back(std::forward<ResourceContainerNameT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The identifier of the image related to a finding.</p>
     */
    inline const Aws::Vector<StringFilter>& GetResourceContainerImageId() const { return m_resourceContainerImageId; }
    inline bool ResourceContainerImageIdHasBeenSet() const { return m_resourceContainerImageIdHasBeenSet; }
    template<typename ResourceContainerImageIdT = Aws::Vector<StringFilter>>
    void SetResourceContainerImageId(ResourceContainerImageIdT&& value) { m_resourceContainerImageIdHasBeenSet = true; m_resourceContainerImageId = std::forward<ResourceContainerImageIdT>(value); }
    template<typename ResourceContainerImageIdT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithResourceContainerImageId(ResourceContainerImageIdT&& value) { SetResourceContainerImageId(std::forward<ResourceContainerImageIdT>(value)); return *this;}
    template<typename ResourceContainerImageIdT = StringFilter>
    AwsSecurityFindingFilters& AddResourceContainerImageId(ResourceContainerImageIdT&& value) { m_resourceContainerImageIdHasBeenSet = true; m_resourceContainerImageId.emplace_back(std::forward<ResourceContainerImageIdT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The name of the image related to a finding.</p>
     */
    inline const Aws::Vector<StringFilter>& GetResourceContainerImageName() const { return m_resourceContainerImageName; }
    inline bool ResourceContainerImageNameHasBeenSet() const { return m_resourceContainerImageNameHasBeenSet; }
    template<typename ResourceContainerImageNameT = Aws::Vector<StringFilter>>
    void SetResourceContainerImageName(ResourceContainerImageNameT&& value) { m_resourceContainerImageNameHasBeenSet = true; m_resourceContainerImageName = std::forward<ResourceContainerImageNameT>(value); }
    template<typename ResourceContainerImageNameT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithResourceContainerImageName(ResourceContainerImageNameT&& value) { SetResourceContainerImageName(std::forward<ResourceContainerImageNameT>(value)); return *this;}
    template<typename ResourceContainerImageNameT = StringFilter>
    AwsSecurityFindingFilters& AddResourceContainerImageName(ResourceContainerImageNameT&& value) { m_resourceContainerImageNameHasBeenSet = true; m_resourceContainerImageName.emplace_back(std::forward<ResourceContainerImageNameT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>A timestamp that identifies when the container was started.</p> <p>For more
     * information about the validation and formatting of timestamp fields in Security
     * Hub, see <a
     * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/Welcome.html#timestamps">Timestamps</a>.</p>
     */
    inline const Aws::Vector<DateFilter>& GetResourceContainerLaunchedAt() const { return m_resourceContainerLaunchedAt; }
    inline bool ResourceContainerLaunchedAtHasBeenSet() const { return m_resourceContainerLaunchedAtHasBeenSet; }
    template<typename ResourceContainerLaunchedAtT = Aws::Vector<DateFilter>>
    void SetResourceContainerLaunchedAt(ResourceContainerLaunchedAtT&& value) { m_resourceContainerLaunchedAtHasBeenSet = true; m_resourceContainerLaunchedAt = std::forward<ResourceContainerLaunchedAtT>(value); }
    template<typename ResourceContainerLaunchedAtT = Aws::Vector<DateFilter>>
    AwsSecurityFindingFilters& WithResourceContainerLaunchedAt(ResourceContainerLaunchedAtT&& value) { SetResourceContainerLaunchedAt(std::forward<ResourceContainerLaunchedAtT>(value)); return *this;}
    template<typename ResourceContainerLaunchedAtT = DateFilter>
    AwsSecurityFindingFilters& AddResourceContainerLaunchedAt(ResourceContainerLaunchedAtT&& value) { m_resourceContainerLaunchedAtHasBeenSet = true; m_resourceContainerLaunchedAt.emplace_back(std::forward<ResourceContainerLaunchedAtT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The details of a resource that doesn't have a specific subfield for the
     * resource type defined.</p>
     */
    inline const Aws::Vector<MapFilter>& GetResourceDetailsOther() const { return m_resourceDetailsOther; }
    inline bool ResourceDetailsOtherHasBeenSet() const { return m_resourceDetailsOtherHasBeenSet; }
    template<typename ResourceDetailsOtherT = Aws::Vector<MapFilter>>
    void SetResourceDetailsOther(ResourceDetailsOtherT&& value) { m_resourceDetailsOtherHasBeenSet = true; m_resourceDetailsOther = std::forward<ResourceDetailsOtherT>(value); }
    template<typename ResourceDetailsOtherT = Aws::Vector<MapFilter>>
    AwsSecurityFindingFilters& WithResourceDetailsOther(ResourceDetailsOtherT&& value) { SetResourceDetailsOther(std::forward<ResourceDetailsOtherT>(value)); return *this;}
    template<typename ResourceDetailsOtherT = MapFilter>
    AwsSecurityFindingFilters& AddResourceDetailsOther(ResourceDetailsOtherT&& value) { m_resourceDetailsOtherHasBeenSet = true; m_resourceDetailsOther.emplace_back(std::forward<ResourceDetailsOtherT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>Exclusive to findings that are generated as the result of a check run against
     * a specific rule in a supported standard, such as CIS Amazon Web Services
     * Foundations. Contains security standard-related finding details.</p>
     */
    inline const Aws::Vector<StringFilter>& GetComplianceStatus() const { return m_complianceStatus; }
    inline bool ComplianceStatusHasBeenSet() const { return m_complianceStatusHasBeenSet; }
    template<typename ComplianceStatusT = Aws::Vector<StringFilter>>
    void SetComplianceStatus(ComplianceStatusT&& value) { m_complianceStatusHasBeenSet = true; m_complianceStatus = std::forward<ComplianceStatusT>(value); }
    template<typename ComplianceStatusT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithComplianceStatus(ComplianceStatusT&& value) { SetComplianceStatus(std::forward<ComplianceStatusT>(value)); return *this;}
    template<typename ComplianceStatusT = StringFilter>
    AwsSecurityFindingFilters& AddComplianceStatus(ComplianceStatusT&& value) { m_complianceStatusHasBeenSet = true; m_complianceStatus.emplace_back(std::forward<ComplianceStatusT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The veracity of a finding.</p>
     */
    inline const Aws::Vector<StringFilter>& GetVerificationState() const { return m_verificationState; }
    inline bool VerificationStateHasBeenSet() const { return m_verificationStateHasBeenSet; }
    template<typename VerificationStateT = Aws::Vector<StringFilter>>
    void SetVerificationState(VerificationStateT&& value) { m_verificationStateHasBeenSet = true; m_verificationState = std::forward<VerificationStateT>(value); }
    template<typename VerificationStateT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithVerificationState(VerificationStateT&& value) { SetVerificationState(std::forward<VerificationStateT>(value)); return *this;}
    template<typename VerificationStateT = StringFilter>
    AwsSecurityFindingFilters& AddVerificationState(VerificationStateT&& value) { m_verificationStateHasBeenSet = true; m_verificationState.emplace_back(std::forward<VerificationStateT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The workflow state of a finding.</p> <p>Note that this field is deprecated.
     * To search for a finding based on its workflow status, use
     * <code>WorkflowStatus</code>.</p>
     */
    inline const Aws::Vector<StringFilter>& GetWorkflowState() const { return m_workflowState; }
    inline bool WorkflowStateHasBeenSet() const { return m_workflowStateHasBeenSet; }
    template<typename WorkflowStateT = Aws::Vector<StringFilter>>
    void SetWorkflowState(WorkflowStateT&& value) { m_workflowStateHasBeenSet = true; m_workflowState = std::forward<WorkflowStateT>(value); }
    template<typename WorkflowStateT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithWorkflowState(WorkflowStateT&& value) { SetWorkflowState(std::forward<WorkflowStateT>(value)); return *this;}
    template<typename WorkflowStateT = StringFilter>
    AwsSecurityFindingFilters& AddWorkflowState(WorkflowStateT&& value) { m_workflowStateHasBeenSet = true; m_workflowState.emplace_back(std::forward<WorkflowStateT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The status of the investigation into a finding. Allowed values are the
     * following.</p> <ul> <li> <p> <code>NEW</code> - The initial state of a finding,
     * before it is reviewed.</p> <p>Security Hub also resets the workflow status from
     * <code>NOTIFIED</code> or <code>RESOLVED</code> to <code>NEW</code> in the
     * following cases:</p> <ul> <li> <p> <code>RecordState</code> changes from
     * <code>ARCHIVED</code> to <code>ACTIVE</code>.</p> </li> <li> <p>
     * <code>Compliance.Status</code> changes from <code>PASSED</code> to either
     * <code>WARNING</code>, <code>FAILED</code>, or <code>NOT_AVAILABLE</code>.</p>
     * </li> </ul> </li> <li> <p> <code>NOTIFIED</code> - Indicates that the resource
     * owner has been notified about the security issue. Used when the initial reviewer
     * is not the resource owner, and needs intervention from the resource owner.</p>
     * <p>If one of the following occurs, the workflow status is changed automatically
     * from <code>NOTIFIED</code> to <code>NEW</code>:</p> <ul> <li> <p>
     * <code>RecordState</code> changes from <code>ARCHIVED</code> to
     * <code>ACTIVE</code>.</p> </li> <li> <p> <code>Compliance.Status</code> changes
     * from <code>PASSED</code> to <code>FAILED</code>, <code>WARNING</code>, or
     * <code>NOT_AVAILABLE</code>.</p> </li> </ul> </li> <li> <p>
     * <code>SUPPRESSED</code> - Indicates that you reviewed the finding and don't
     * believe that any action is needed.</p> <p>The workflow status of a
     * <code>SUPPRESSED</code> finding does not change if <code>RecordState</code>
     * changes from <code>ARCHIVED</code> to <code>ACTIVE</code>.</p> </li> <li> <p>
     * <code>RESOLVED</code> - The finding was reviewed and remediated and is now
     * considered resolved. </p> <p>The finding remains <code>RESOLVED</code> unless
     * one of the following occurs:</p> <ul> <li> <p> <code>RecordState</code> changes
     * from <code>ARCHIVED</code> to <code>ACTIVE</code>.</p> </li> <li> <p>
     * <code>Compliance.Status</code> changes from <code>PASSED</code> to
     * <code>FAILED</code>, <code>WARNING</code>, or <code>NOT_AVAILABLE</code>.</p>
     * </li> </ul> <p>In those cases, the workflow status is automatically reset to
     * <code>NEW</code>.</p> <p>For findings from controls, if
     * <code>Compliance.Status</code> is <code>PASSED</code>, then Security Hub
     * automatically sets the workflow status to <code>RESOLVED</code>.</p> </li> </ul>
     */
    inline const Aws::Vector<StringFilter>& GetWorkflowStatus() const { return m_workflowStatus; }
    inline bool WorkflowStatusHasBeenSet() const { return m_workflowStatusHasBeenSet; }
    template<typename WorkflowStatusT = Aws::Vector<StringFilter>>
    void SetWorkflowStatus(WorkflowStatusT&& value) { m_workflowStatusHasBeenSet = true; m_workflowStatus = std::forward<WorkflowStatusT>(value); }
    template<typename WorkflowStatusT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithWorkflowStatus(WorkflowStatusT&& value) { SetWorkflowStatus(std::forward<WorkflowStatusT>(value)); return *this;}
    template<typename WorkflowStatusT = StringFilter>
    AwsSecurityFindingFilters& AddWorkflowStatus(WorkflowStatusT&& value) { m_workflowStatusHasBeenSet = true; m_workflowStatus.emplace_back(std::forward<WorkflowStatusT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The updated record state for the finding.</p>
     */
    inline const Aws::Vector<StringFilter>& GetRecordState() const { return m_recordState; }
    inline bool RecordStateHasBeenSet() const { return m_recordStateHasBeenSet; }
    template<typename RecordStateT = Aws::Vector<StringFilter>>
    void SetRecordState(RecordStateT&& value) { m_recordStateHasBeenSet = true; m_recordState = std::forward<RecordStateT>(value); }
    template<typename RecordStateT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithRecordState(RecordStateT&& value) { SetRecordState(std::forward<RecordStateT>(value)); return *this;}
    template<typename RecordStateT = StringFilter>
    AwsSecurityFindingFilters& AddRecordState(RecordStateT&& value) { m_recordStateHasBeenSet = true; m_recordState.emplace_back(std::forward<RecordStateT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The ARN of the solution that generated a related finding.</p>
     */
    inline const Aws::Vector<StringFilter>& GetRelatedFindingsProductArn() const { return m_relatedFindingsProductArn; }
    inline bool RelatedFindingsProductArnHasBeenSet() const { return m_relatedFindingsProductArnHasBeenSet; }
    template<typename RelatedFindingsProductArnT = Aws::Vector<StringFilter>>
    void SetRelatedFindingsProductArn(RelatedFindingsProductArnT&& value) { m_relatedFindingsProductArnHasBeenSet = true; m_relatedFindingsProductArn = std::forward<RelatedFindingsProductArnT>(value); }
    template<typename RelatedFindingsProductArnT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithRelatedFindingsProductArn(RelatedFindingsProductArnT&& value) { SetRelatedFindingsProductArn(std::forward<RelatedFindingsProductArnT>(value)); return *this;}
    template<typename RelatedFindingsProductArnT = StringFilter>
    AwsSecurityFindingFilters& AddRelatedFindingsProductArn(RelatedFindingsProductArnT&& value) { m_relatedFindingsProductArnHasBeenSet = true; m_relatedFindingsProductArn.emplace_back(std::forward<RelatedFindingsProductArnT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The solution-generated identifier for a related finding.</p>
     */
    inline const Aws::Vector<StringFilter>& GetRelatedFindingsId() const { return m_relatedFindingsId; }
    inline bool RelatedFindingsIdHasBeenSet() const { return m_relatedFindingsIdHasBeenSet; }
    template<typename RelatedFindingsIdT = Aws::Vector<StringFilter>>
    void SetRelatedFindingsId(RelatedFindingsIdT&& value) { m_relatedFindingsIdHasBeenSet = true; m_relatedFindingsId = std::forward<RelatedFindingsIdT>(value); }
    template<typename RelatedFindingsIdT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithRelatedFindingsId(RelatedFindingsIdT&& value) { SetRelatedFindingsId(std::forward<RelatedFindingsIdT>(value)); return *this;}
    template<typename RelatedFindingsIdT = StringFilter>
    AwsSecurityFindingFilters& AddRelatedFindingsId(RelatedFindingsIdT&& value) { m_relatedFindingsIdHasBeenSet = true; m_relatedFindingsId.emplace_back(std::forward<RelatedFindingsIdT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The text of a note.</p>
     */
    inline const Aws::Vector<StringFilter>& GetNoteText() const { return m_noteText; }
    inline bool NoteTextHasBeenSet() const { return m_noteTextHasBeenSet; }
    template<typename NoteTextT = Aws::Vector<StringFilter>>
    void SetNoteText(NoteTextT&& value) { m_noteTextHasBeenSet = true; m_noteText = std::forward<NoteTextT>(value); }
    template<typename NoteTextT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithNoteText(NoteTextT&& value) { SetNoteText(std::forward<NoteTextT>(value)); return *this;}
    template<typename NoteTextT = StringFilter>
    AwsSecurityFindingFilters& AddNoteText(NoteTextT&& value) { m_noteTextHasBeenSet = true; m_noteText.emplace_back(std::forward<NoteTextT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The timestamp of when the note was updated.</p>
     */
    inline const Aws::Vector<DateFilter>& GetNoteUpdatedAt() const { return m_noteUpdatedAt; }
    inline bool NoteUpdatedAtHasBeenSet() const { return m_noteUpdatedAtHasBeenSet; }
    template<typename NoteUpdatedAtT = Aws::Vector<DateFilter>>
    void SetNoteUpdatedAt(NoteUpdatedAtT&& value) { m_noteUpdatedAtHasBeenSet = true; m_noteUpdatedAt = std::forward<NoteUpdatedAtT>(value); }
    template<typename NoteUpdatedAtT = Aws::Vector<DateFilter>>
    AwsSecurityFindingFilters& WithNoteUpdatedAt(NoteUpdatedAtT&& value) { SetNoteUpdatedAt(std::forward<NoteUpdatedAtT>(value)); return *this;}
    template<typename NoteUpdatedAtT = DateFilter>
    AwsSecurityFindingFilters& AddNoteUpdatedAt(NoteUpdatedAtT&& value) { m_noteUpdatedAtHasBeenSet = true; m_noteUpdatedAt.emplace_back(std::forward<NoteUpdatedAtT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The principal that created a note.</p>
     */
    inline const Aws::Vector<StringFilter>& GetNoteUpdatedBy() const { return m_noteUpdatedBy; }
    inline bool NoteUpdatedByHasBeenSet() const { return m_noteUpdatedByHasBeenSet; }
    template<typename NoteUpdatedByT = Aws::Vector<StringFilter>>
    void SetNoteUpdatedBy(NoteUpdatedByT&& value) { m_noteUpdatedByHasBeenSet = true; m_noteUpdatedBy = std::forward<NoteUpdatedByT>(value); }
    template<typename NoteUpdatedByT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithNoteUpdatedBy(NoteUpdatedByT&& value) { SetNoteUpdatedBy(std::forward<NoteUpdatedByT>(value)); return *this;}
    template<typename NoteUpdatedByT = StringFilter>
    AwsSecurityFindingFilters& AddNoteUpdatedBy(NoteUpdatedByT&& value) { m_noteUpdatedByHasBeenSet = true; m_noteUpdatedBy.emplace_back(std::forward<NoteUpdatedByT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The finding provider value for the finding confidence. Confidence is defined
     * as the likelihood that a finding accurately identifies the behavior or issue
     * that it was intended to identify.</p> <p>Confidence is scored on a 0-100 basis
     * using a ratio scale, where 0 means zero percent confidence and 100 means 100
     * percent confidence.</p>
     */
    inline const Aws::Vector<NumberFilter>& GetFindingProviderFieldsConfidence() const { return m_findingProviderFieldsConfidence; }
    inline bool FindingProviderFieldsConfidenceHasBeenSet() const { return m_findingProviderFieldsConfidenceHasBeenSet; }
    template<typename FindingProviderFieldsConfidenceT = Aws::Vector<NumberFilter>>
    void SetFindingProviderFieldsConfidence(FindingProviderFieldsConfidenceT&& value) { m_findingProviderFieldsConfidenceHasBeenSet = true; m_findingProviderFieldsConfidence = std::forward<FindingProviderFieldsConfidenceT>(value); }
    template<typename FindingProviderFieldsConfidenceT = Aws::Vector<NumberFilter>>
    AwsSecurityFindingFilters& WithFindingProviderFieldsConfidence(FindingProviderFieldsConfidenceT&& value) { SetFindingProviderFieldsConfidence(std::forward<FindingProviderFieldsConfidenceT>(value)); return *this;}
    template<typename FindingProviderFieldsConfidenceT = NumberFilter>
    AwsSecurityFindingFilters& AddFindingProviderFieldsConfidence(FindingProviderFieldsConfidenceT&& value) { m_findingProviderFieldsConfidenceHasBeenSet = true; m_findingProviderFieldsConfidence.emplace_back(std::forward<FindingProviderFieldsConfidenceT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The finding provider value for the level of importance assigned to the
     * resources associated with the findings.</p> <p>A score of 0 means that the
     * underlying resources have no criticality, and a score of 100 is reserved for the
     * most critical resources. </p>
     */
    inline const Aws::Vector<NumberFilter>& GetFindingProviderFieldsCriticality() const { return m_findingProviderFieldsCriticality; }
    inline bool FindingProviderFieldsCriticalityHasBeenSet() const { return m_findingProviderFieldsCriticalityHasBeenSet; }
    template<typename FindingProviderFieldsCriticalityT = Aws::Vector<NumberFilter>>
    void SetFindingProviderFieldsCriticality(FindingProviderFieldsCriticalityT&& value) { m_findingProviderFieldsCriticalityHasBeenSet = true; m_findingProviderFieldsCriticality = std::forward<FindingProviderFieldsCriticalityT>(value); }
    template<typename FindingProviderFieldsCriticalityT = Aws::Vector<NumberFilter>>
    AwsSecurityFindingFilters& WithFindingProviderFieldsCriticality(FindingProviderFieldsCriticalityT&& value) { SetFindingProviderFieldsCriticality(std::forward<FindingProviderFieldsCriticalityT>(value)); return *this;}
    template<typename FindingProviderFieldsCriticalityT = NumberFilter>
    AwsSecurityFindingFilters& AddFindingProviderFieldsCriticality(FindingProviderFieldsCriticalityT&& value) { m_findingProviderFieldsCriticalityHasBeenSet = true; m_findingProviderFieldsCriticality.emplace_back(std::forward<FindingProviderFieldsCriticalityT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The finding identifier of a related finding that is identified by the finding
     * provider.</p>
     */
    inline const Aws::Vector<StringFilter>& GetFindingProviderFieldsRelatedFindingsId() const { return m_findingProviderFieldsRelatedFindingsId; }
    inline bool FindingProviderFieldsRelatedFindingsIdHasBeenSet() const { return m_findingProviderFieldsRelatedFindingsIdHasBeenSet; }
    template<typename FindingProviderFieldsRelatedFindingsIdT = Aws::Vector<StringFilter>>
    void SetFindingProviderFieldsRelatedFindingsId(FindingProviderFieldsRelatedFindingsIdT&& value) { m_findingProviderFieldsRelatedFindingsIdHasBeenSet = true; m_findingProviderFieldsRelatedFindingsId = std::forward<FindingProviderFieldsRelatedFindingsIdT>(value); }
    template<typename FindingProviderFieldsRelatedFindingsIdT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithFindingProviderFieldsRelatedFindingsId(FindingProviderFieldsRelatedFindingsIdT&& value) { SetFindingProviderFieldsRelatedFindingsId(std::forward<FindingProviderFieldsRelatedFindingsIdT>(value)); return *this;}
    template<typename FindingProviderFieldsRelatedFindingsIdT = StringFilter>
    AwsSecurityFindingFilters& AddFindingProviderFieldsRelatedFindingsId(FindingProviderFieldsRelatedFindingsIdT&& value) { m_findingProviderFieldsRelatedFindingsIdHasBeenSet = true; m_findingProviderFieldsRelatedFindingsId.emplace_back(std::forward<FindingProviderFieldsRelatedFindingsIdT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The ARN of the solution that generated a related finding that is identified
     * by the finding provider.</p>
     */
    inline const Aws::Vector<StringFilter>& GetFindingProviderFieldsRelatedFindingsProductArn() const { return m_findingProviderFieldsRelatedFindingsProductArn; }
    inline bool FindingProviderFieldsRelatedFindingsProductArnHasBeenSet() const { return m_findingProviderFieldsRelatedFindingsProductArnHasBeenSet; }
    template<typename FindingProviderFieldsRelatedFindingsProductArnT = Aws::Vector<StringFilter>>
    void SetFindingProviderFieldsRelatedFindingsProductArn(FindingProviderFieldsRelatedFindingsProductArnT&& value) { m_findingProviderFieldsRelatedFindingsProductArnHasBeenSet = true; m_findingProviderFieldsRelatedFindingsProductArn = std::forward<FindingProviderFieldsRelatedFindingsProductArnT>(value); }
    template<typename FindingProviderFieldsRelatedFindingsProductArnT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithFindingProviderFieldsRelatedFindingsProductArn(FindingProviderFieldsRelatedFindingsProductArnT&& value) { SetFindingProviderFieldsRelatedFindingsProductArn(std::forward<FindingProviderFieldsRelatedFindingsProductArnT>(value)); return *this;}
    template<typename FindingProviderFieldsRelatedFindingsProductArnT = StringFilter>
    AwsSecurityFindingFilters& AddFindingProviderFieldsRelatedFindingsProductArn(FindingProviderFieldsRelatedFindingsProductArnT&& value) { m_findingProviderFieldsRelatedFindingsProductArnHasBeenSet = true; m_findingProviderFieldsRelatedFindingsProductArn.emplace_back(std::forward<FindingProviderFieldsRelatedFindingsProductArnT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The finding provider value for the severity label.</p>
     */
    inline const Aws::Vector<StringFilter>& GetFindingProviderFieldsSeverityLabel() const { return m_findingProviderFieldsSeverityLabel; }
    inline bool FindingProviderFieldsSeverityLabelHasBeenSet() const { return m_findingProviderFieldsSeverityLabelHasBeenSet; }
    template<typename FindingProviderFieldsSeverityLabelT = Aws::Vector<StringFilter>>
    void SetFindingProviderFieldsSeverityLabel(FindingProviderFieldsSeverityLabelT&& value) { m_findingProviderFieldsSeverityLabelHasBeenSet = true; m_findingProviderFieldsSeverityLabel = std::forward<FindingProviderFieldsSeverityLabelT>(value); }
    template<typename FindingProviderFieldsSeverityLabelT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithFindingProviderFieldsSeverityLabel(FindingProviderFieldsSeverityLabelT&& value) { SetFindingProviderFieldsSeverityLabel(std::forward<FindingProviderFieldsSeverityLabelT>(value)); return *this;}
    template<typename FindingProviderFieldsSeverityLabelT = StringFilter>
    AwsSecurityFindingFilters& AddFindingProviderFieldsSeverityLabel(FindingProviderFieldsSeverityLabelT&& value) { m_findingProviderFieldsSeverityLabelHasBeenSet = true; m_findingProviderFieldsSeverityLabel.emplace_back(std::forward<FindingProviderFieldsSeverityLabelT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The finding provider's original value for the severity.</p>
     */
    inline const Aws::Vector<StringFilter>& GetFindingProviderFieldsSeverityOriginal() const { return m_findingProviderFieldsSeverityOriginal; }
    inline bool FindingProviderFieldsSeverityOriginalHasBeenSet() const { return m_findingProviderFieldsSeverityOriginalHasBeenSet; }
    template<typename FindingProviderFieldsSeverityOriginalT = Aws::Vector<StringFilter>>
    void SetFindingProviderFieldsSeverityOriginal(FindingProviderFieldsSeverityOriginalT&& value) { m_findingProviderFieldsSeverityOriginalHasBeenSet = true; m_findingProviderFieldsSeverityOriginal = std::forward<FindingProviderFieldsSeverityOriginalT>(value); }
    template<typename FindingProviderFieldsSeverityOriginalT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithFindingProviderFieldsSeverityOriginal(FindingProviderFieldsSeverityOriginalT&& value) { SetFindingProviderFieldsSeverityOriginal(std::forward<FindingProviderFieldsSeverityOriginalT>(value)); return *this;}
    template<typename FindingProviderFieldsSeverityOriginalT = StringFilter>
    AwsSecurityFindingFilters& AddFindingProviderFieldsSeverityOriginal(FindingProviderFieldsSeverityOriginalT&& value) { m_findingProviderFieldsSeverityOriginalHasBeenSet = true; m_findingProviderFieldsSeverityOriginal.emplace_back(std::forward<FindingProviderFieldsSeverityOriginalT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>One or more finding types that the finding provider assigned to the finding.
     * Uses the format of <code>namespace/category/classifier</code> that classify a
     * finding.</p> <p>Valid namespace values are: Software and Configuration Checks |
     * TTPs | Effects | Unusual Behaviors | Sensitive Data Identifications</p>
     */
    inline const Aws::Vector<StringFilter>& GetFindingProviderFieldsTypes() const { return m_findingProviderFieldsTypes; }
    inline bool FindingProviderFieldsTypesHasBeenSet() const { return m_findingProviderFieldsTypesHasBeenSet; }
    template<typename FindingProviderFieldsTypesT = Aws::Vector<StringFilter>>
    void SetFindingProviderFieldsTypes(FindingProviderFieldsTypesT&& value) { m_findingProviderFieldsTypesHasBeenSet = true; m_findingProviderFieldsTypes = std::forward<FindingProviderFieldsTypesT>(value); }
    template<typename FindingProviderFieldsTypesT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithFindingProviderFieldsTypes(FindingProviderFieldsTypesT&& value) { SetFindingProviderFieldsTypes(std::forward<FindingProviderFieldsTypesT>(value)); return *this;}
    template<typename FindingProviderFieldsTypesT = StringFilter>
    AwsSecurityFindingFilters& AddFindingProviderFieldsTypes(FindingProviderFieldsTypesT&& value) { m_findingProviderFieldsTypesHasBeenSet = true; m_findingProviderFieldsTypes.emplace_back(std::forward<FindingProviderFieldsTypesT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>Indicates whether or not sample findings are included in the filter
     * results.</p>
     */
    inline const Aws::Vector<BooleanFilter>& GetSample() const { return m_sample; }
    inline bool SampleHasBeenSet() const { return m_sampleHasBeenSet; }
    template<typename SampleT = Aws::Vector<BooleanFilter>>
    void SetSample(SampleT&& value) { m_sampleHasBeenSet = true; m_sample = std::forward<SampleT>(value); }
    template<typename SampleT = Aws::Vector<BooleanFilter>>
    AwsSecurityFindingFilters& WithSample(SampleT&& value) { SetSample(std::forward<SampleT>(value)); return *this;}
    template<typename SampleT = BooleanFilter>
    AwsSecurityFindingFilters& AddSample(SampleT&& value) { m_sampleHasBeenSet = true; m_sample.emplace_back(std::forward<SampleT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p> The unique identifier of a control across standards. Values for this field
     * typically consist of an Amazon Web Services service and a number, such as
     * APIGateway.5. </p>
     */
    inline const Aws::Vector<StringFilter>& GetComplianceSecurityControlId() const { return m_complianceSecurityControlId; }
    inline bool ComplianceSecurityControlIdHasBeenSet() const { return m_complianceSecurityControlIdHasBeenSet; }
    template<typename ComplianceSecurityControlIdT = Aws::Vector<StringFilter>>
    void SetComplianceSecurityControlId(ComplianceSecurityControlIdT&& value) { m_complianceSecurityControlIdHasBeenSet = true; m_complianceSecurityControlId = std::forward<ComplianceSecurityControlIdT>(value); }
    template<typename ComplianceSecurityControlIdT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithComplianceSecurityControlId(ComplianceSecurityControlIdT&& value) { SetComplianceSecurityControlId(std::forward<ComplianceSecurityControlIdT>(value)); return *this;}
    template<typename ComplianceSecurityControlIdT = StringFilter>
    AwsSecurityFindingFilters& AddComplianceSecurityControlId(ComplianceSecurityControlIdT&& value) { m_complianceSecurityControlIdHasBeenSet = true; m_complianceSecurityControlId.emplace_back(std::forward<ComplianceSecurityControlIdT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p> The unique identifier of a standard in which a control is enabled. This
     * field consists of the resource portion of the Amazon Resource Name (ARN)
     * returned for a standard in the <a
     * href="https://docs.aws.amazon.com/securityhub/1.0/APIReference/API_DescribeStandards.html">DescribeStandards</a>
     * API response. </p>
     */
    inline const Aws::Vector<StringFilter>& GetComplianceAssociatedStandardsId() const { return m_complianceAssociatedStandardsId; }
    inline bool ComplianceAssociatedStandardsIdHasBeenSet() const { return m_complianceAssociatedStandardsIdHasBeenSet; }
    template<typename ComplianceAssociatedStandardsIdT = Aws::Vector<StringFilter>>
    void SetComplianceAssociatedStandardsId(ComplianceAssociatedStandardsIdT&& value) { m_complianceAssociatedStandardsIdHasBeenSet = true; m_complianceAssociatedStandardsId = std::forward<ComplianceAssociatedStandardsIdT>(value); }
    template<typename ComplianceAssociatedStandardsIdT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithComplianceAssociatedStandardsId(ComplianceAssociatedStandardsIdT&& value) { SetComplianceAssociatedStandardsId(std::forward<ComplianceAssociatedStandardsIdT>(value)); return *this;}
    template<typename ComplianceAssociatedStandardsIdT = StringFilter>
    AwsSecurityFindingFilters& AddComplianceAssociatedStandardsId(ComplianceAssociatedStandardsIdT&& value) { m_complianceAssociatedStandardsIdHasBeenSet = true; m_complianceAssociatedStandardsId.emplace_back(std::forward<ComplianceAssociatedStandardsIdT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p> Indicates whether a software vulnerability in your environment has a known
     * exploit. You can filter findings by this field only if you use Security Hub and
     * Amazon Inspector. </p>
     */
    inline const Aws::Vector<StringFilter>& GetVulnerabilitiesExploitAvailable() const { return m_vulnerabilitiesExploitAvailable; }
    inline bool VulnerabilitiesExploitAvailableHasBeenSet() const { return m_vulnerabilitiesExploitAvailableHasBeenSet; }
    template<typename VulnerabilitiesExploitAvailableT = Aws::Vector<StringFilter>>
    void SetVulnerabilitiesExploitAvailable(VulnerabilitiesExploitAvailableT&& value) { m_vulnerabilitiesExploitAvailableHasBeenSet = true; m_vulnerabilitiesExploitAvailable = std::forward<VulnerabilitiesExploitAvailableT>(value); }
    template<typename VulnerabilitiesExploitAvailableT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithVulnerabilitiesExploitAvailable(VulnerabilitiesExploitAvailableT&& value) { SetVulnerabilitiesExploitAvailable(std::forward<VulnerabilitiesExploitAvailableT>(value)); return *this;}
    template<typename VulnerabilitiesExploitAvailableT = StringFilter>
    AwsSecurityFindingFilters& AddVulnerabilitiesExploitAvailable(VulnerabilitiesExploitAvailableT&& value) { m_vulnerabilitiesExploitAvailableHasBeenSet = true; m_vulnerabilitiesExploitAvailable.emplace_back(std::forward<VulnerabilitiesExploitAvailableT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p> Indicates whether a vulnerability is fixed in a newer version of the
     * affected software packages. You can filter findings by this field only if you
     * use Security Hub and Amazon Inspector. </p>
     */
    inline const Aws::Vector<StringFilter>& GetVulnerabilitiesFixAvailable() const { return m_vulnerabilitiesFixAvailable; }
    inline bool VulnerabilitiesFixAvailableHasBeenSet() const { return m_vulnerabilitiesFixAvailableHasBeenSet; }
    template<typename VulnerabilitiesFixAvailableT = Aws::Vector<StringFilter>>
    void SetVulnerabilitiesFixAvailable(VulnerabilitiesFixAvailableT&& value) { m_vulnerabilitiesFixAvailableHasBeenSet = true; m_vulnerabilitiesFixAvailable = std::forward<VulnerabilitiesFixAvailableT>(value); }
    template<typename VulnerabilitiesFixAvailableT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithVulnerabilitiesFixAvailable(VulnerabilitiesFixAvailableT&& value) { SetVulnerabilitiesFixAvailable(std::forward<VulnerabilitiesFixAvailableT>(value)); return *this;}
    template<typename VulnerabilitiesFixAvailableT = StringFilter>
    AwsSecurityFindingFilters& AddVulnerabilitiesFixAvailable(VulnerabilitiesFixAvailableT&& value) { m_vulnerabilitiesFixAvailableHasBeenSet = true; m_vulnerabilitiesFixAvailable.emplace_back(std::forward<VulnerabilitiesFixAvailableT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p> The name of a security control parameter. </p>
     */
    inline const Aws::Vector<StringFilter>& GetComplianceSecurityControlParametersName() const { return m_complianceSecurityControlParametersName; }
    inline bool ComplianceSecurityControlParametersNameHasBeenSet() const { return m_complianceSecurityControlParametersNameHasBeenSet; }
    template<typename ComplianceSecurityControlParametersNameT = Aws::Vector<StringFilter>>
    void SetComplianceSecurityControlParametersName(ComplianceSecurityControlParametersNameT&& value) { m_complianceSecurityControlParametersNameHasBeenSet = true; m_complianceSecurityControlParametersName = std::forward<ComplianceSecurityControlParametersNameT>(value); }
    template<typename ComplianceSecurityControlParametersNameT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithComplianceSecurityControlParametersName(ComplianceSecurityControlParametersNameT&& value) { SetComplianceSecurityControlParametersName(std::forward<ComplianceSecurityControlParametersNameT>(value)); return *this;}
    template<typename ComplianceSecurityControlParametersNameT = StringFilter>
    AwsSecurityFindingFilters& AddComplianceSecurityControlParametersName(ComplianceSecurityControlParametersNameT&& value) { m_complianceSecurityControlParametersNameHasBeenSet = true; m_complianceSecurityControlParametersName.emplace_back(std::forward<ComplianceSecurityControlParametersNameT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p> The current value of a security control parameter. </p>
     */
    inline const Aws::Vector<StringFilter>& GetComplianceSecurityControlParametersValue() const { return m_complianceSecurityControlParametersValue; }
    inline bool ComplianceSecurityControlParametersValueHasBeenSet() const { return m_complianceSecurityControlParametersValueHasBeenSet; }
    template<typename ComplianceSecurityControlParametersValueT = Aws::Vector<StringFilter>>
    void SetComplianceSecurityControlParametersValue(ComplianceSecurityControlParametersValueT&& value) { m_complianceSecurityControlParametersValueHasBeenSet = true; m_complianceSecurityControlParametersValue = std::forward<ComplianceSecurityControlParametersValueT>(value); }
    template<typename ComplianceSecurityControlParametersValueT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithComplianceSecurityControlParametersValue(ComplianceSecurityControlParametersValueT&& value) { SetComplianceSecurityControlParametersValue(std::forward<ComplianceSecurityControlParametersValueT>(value)); return *this;}
    template<typename ComplianceSecurityControlParametersValueT = StringFilter>
    AwsSecurityFindingFilters& AddComplianceSecurityControlParametersValue(ComplianceSecurityControlParametersValueT&& value) { m_complianceSecurityControlParametersValueHasBeenSet = true; m_complianceSecurityControlParametersValue.emplace_back(std::forward<ComplianceSecurityControlParametersValueT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p>The name of the Amazon Web Services account in which a finding is
     * generated.</p>
     */
    inline const Aws::Vector<StringFilter>& GetAwsAccountName() const { return m_awsAccountName; }
    inline bool AwsAccountNameHasBeenSet() const { return m_awsAccountNameHasBeenSet; }
    template<typename AwsAccountNameT = Aws::Vector<StringFilter>>
    void SetAwsAccountName(AwsAccountNameT&& value) { m_awsAccountNameHasBeenSet = true; m_awsAccountName = std::forward<AwsAccountNameT>(value); }
    template<typename AwsAccountNameT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithAwsAccountName(AwsAccountNameT&& value) { SetAwsAccountName(std::forward<AwsAccountNameT>(value)); return *this;}
    template<typename AwsAccountNameT = StringFilter>
    AwsSecurityFindingFilters& AddAwsAccountName(AwsAccountNameT&& value) { m_awsAccountNameHasBeenSet = true; m_awsAccountName.emplace_back(std::forward<AwsAccountNameT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p> The name of the application that is related to a finding. </p>
     */
    inline const Aws::Vector<StringFilter>& GetResourceApplicationName() const { return m_resourceApplicationName; }
    inline bool ResourceApplicationNameHasBeenSet() const { return m_resourceApplicationNameHasBeenSet; }
    template<typename ResourceApplicationNameT = Aws::Vector<StringFilter>>
    void SetResourceApplicationName(ResourceApplicationNameT&& value) { m_resourceApplicationNameHasBeenSet = true; m_resourceApplicationName = std::forward<ResourceApplicationNameT>(value); }
    template<typename ResourceApplicationNameT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithResourceApplicationName(ResourceApplicationNameT&& value) { SetResourceApplicationName(std::forward<ResourceApplicationNameT>(value)); return *this;}
    template<typename ResourceApplicationNameT = StringFilter>
    AwsSecurityFindingFilters& AddResourceApplicationName(ResourceApplicationNameT&& value) { m_resourceApplicationNameHasBeenSet = true; m_resourceApplicationName.emplace_back(std::forward<ResourceApplicationNameT>(value)); return *this; }
    ///@}

    ///@{
    /**
     * <p> The ARN of the application that is related to a finding. </p>
     */
    inline const Aws::Vector<StringFilter>& GetResourceApplicationArn() const { return m_resourceApplicationArn; }
    inline bool ResourceApplicationArnHasBeenSet() const { return m_resourceApplicationArnHasBeenSet; }
    template<typename ResourceApplicationArnT = Aws::Vector<StringFilter>>
    void SetResourceApplicationArn(ResourceApplicationArnT&& value) { m_resourceApplicationArnHasBeenSet = true; m_resourceApplicationArn = std::forward<ResourceApplicationArnT>(value); }
    template<typename ResourceApplicationArnT = Aws::Vector<StringFilter>>
    AwsSecurityFindingFilters& WithResourceApplicationArn(ResourceApplicationArnT&& value) { SetResourceApplicationArn(std::forward<ResourceApplicationArnT>(value)); return *this;}
    template<typename ResourceApplicationArnT = StringFilter>
    AwsSecurityFindingFilters& AddResourceApplicationArn(ResourceApplicationArnT&& value) { m_resourceApplicationArnHasBeenSet = true; m_resourceApplicationArn.emplace_back(std::forward<ResourceApplicationArnT>(value)); return *this; }
    ///@}
  private:

    Aws::Vector<StringFilter> m_productArn;
    bool m_productArnHasBeenSet = false;

    Aws::Vector<StringFilter> m_awsAccountId;
    bool m_awsAccountIdHasBeenSet = false;

    Aws::Vector<StringFilter> m_id;
    bool m_idHasBeenSet = false;

    Aws::Vector<StringFilter> m_generatorId;
    bool m_generatorIdHasBeenSet = false;

    Aws::Vector<StringFilter> m_region;
    bool m_regionHasBeenSet = false;

    Aws::Vector<StringFilter> m_type;
    bool m_typeHasBeenSet = false;

    Aws::Vector<DateFilter> m_firstObservedAt;
    bool m_firstObservedAtHasBeenSet = false;

    Aws::Vector<DateFilter> m_lastObservedAt;
    bool m_lastObservedAtHasBeenSet = false;

    Aws::Vector<DateFilter> m_createdAt;
    bool m_createdAtHasBeenSet = false;

    Aws::Vector<DateFilter> m_updatedAt;
    bool m_updatedAtHasBeenSet = false;

    Aws::Vector<StringFilter> m_severityLabel;
    bool m_severityLabelHasBeenSet = false;

    Aws::Vector<NumberFilter> m_confidence;
    bool m_confidenceHasBeenSet = false;

    Aws::Vector<NumberFilter> m_criticality;
    bool m_criticalityHasBeenSet = false;

    Aws::Vector<StringFilter> m_title;
    bool m_titleHasBeenSet = false;

    Aws::Vector<StringFilter> m_description;
    bool m_descriptionHasBeenSet = false;

    Aws::Vector<StringFilter> m_recommendationText;
    bool m_recommendationTextHasBeenSet = false;

    Aws::Vector<StringFilter> m_sourceUrl;
    bool m_sourceUrlHasBeenSet = false;

    Aws::Vector<MapFilter> m_productFields;
    bool m_productFieldsHasBeenSet = false;

    Aws::Vector<StringFilter> m_productName;
    bool m_productNameHasBeenSet = false;

    Aws::Vector<StringFilter> m_companyName;
    bool m_companyNameHasBeenSet = false;

    Aws::Vector<MapFilter> m_userDefinedFields;
    bool m_userDefinedFieldsHasBeenSet = false;

    Aws::Vector<StringFilter> m_malwareName;
    bool m_malwareNameHasBeenSet = false;

    Aws::Vector<StringFilter> m_malwareType;
    bool m_malwareTypeHasBeenSet = false;

    Aws::Vector<StringFilter> m_malwarePath;
    bool m_malwarePathHasBeenSet = false;

    Aws::Vector<StringFilter> m_malwareState;
    bool m_malwareStateHasBeenSet = false;

    Aws::Vector<StringFilter> m_networkDirection;
    bool m_networkDirectionHasBeenSet = false;

    Aws::Vector<StringFilter> m_networkProtocol;
    bool m_networkProtocolHasBeenSet = false;

    Aws::Vector<IpFilter> m_networkSourceIpV4;
    bool m_networkSourceIpV4HasBeenSet = false;

    Aws::Vector<IpFilter> m_networkSourceIpV6;
    bool m_networkSourceIpV6HasBeenSet = false;

    Aws::Vector<NumberFilter> m_networkSourcePort;
    bool m_networkSourcePortHasBeenSet = false;

    Aws::Vector<StringFilter> m_networkSourceDomain;
    bool m_networkSourceDomainHasBeenSet = false;

    Aws::Vector<StringFilter> m_networkSourceMac;
    bool m_networkSourceMacHasBeenSet = false;

    Aws::Vector<IpFilter> m_networkDestinationIpV4;
    bool m_networkDestinationIpV4HasBeenSet = false;

    Aws::Vector<IpFilter> m_networkDestinationIpV6;
    bool m_networkDestinationIpV6HasBeenSet = false;

    Aws::Vector<NumberFilter> m_networkDestinationPort;
    bool m_networkDestinationPortHasBeenSet = false;

    Aws::Vector<StringFilter> m_networkDestinationDomain;
    bool m_networkDestinationDomainHasBeenSet = false;

    Aws::Vector<StringFilter> m_processName;
    bool m_processNameHasBeenSet = false;

    Aws::Vector<StringFilter> m_processPath;
    bool m_processPathHasBeenSet = false;

    Aws::Vector<NumberFilter> m_processPid;
    bool m_processPidHasBeenSet = false;

    Aws::Vector<NumberFilter> m_processParentPid;
    bool m_processParentPidHasBeenSet = false;

    Aws::Vector<DateFilter> m_processLaunchedAt;
    bool m_processLaunchedAtHasBeenSet = false;

    Aws::Vector<DateFilter> m_processTerminatedAt;
    bool m_processTerminatedAtHasBeenSet = false;

    Aws::Vector<StringFilter> m_threatIntelIndicatorType;
    bool m_threatIntelIndicatorTypeHasBeenSet = false;

    Aws::Vector<StringFilter> m_threatIntelIndicatorValue;
    bool m_threatIntelIndicatorValueHasBeenSet = false;

    Aws::Vector<StringFilter> m_threatIntelIndicatorCategory;
    bool m_threatIntelIndicatorCategoryHasBeenSet = false;

    Aws::Vector<DateFilter> m_threatIntelIndicatorLastObservedAt;
    bool m_threatIntelIndicatorLastObservedAtHasBeenSet = false;

    Aws::Vector<StringFilter> m_threatIntelIndicatorSource;
    bool m_threatIntelIndicatorSourceHasBeenSet = false;

    Aws::Vector<StringFilter> m_threatIntelIndicatorSourceUrl;
    bool m_threatIntelIndicatorSourceUrlHasBeenSet = false;

    Aws::Vector<StringFilter> m_resourceType;
    bool m_resourceTypeHasBeenSet = false;

    Aws::Vector<StringFilter> m_resourceId;
    bool m_resourceIdHasBeenSet = false;

    Aws::Vector<StringFilter> m_resourcePartition;
    bool m_resourcePartitionHasBeenSet = false;

    Aws::Vector<StringFilter> m_resourceRegion;
    bool m_resourceRegionHasBeenSet = false;

    Aws::Vector<MapFilter> m_resourceTags;
    bool m_resourceTagsHasBeenSet = false;

    Aws::Vector<StringFilter> m_resourceAwsEc2InstanceType;
    bool m_resourceAwsEc2InstanceTypeHasBeenSet = false;

    Aws::Vector<StringFilter> m_resourceAwsEc2InstanceImageId;
    bool m_resourceAwsEc2InstanceImageIdHasBeenSet = false;

    Aws::Vector<IpFilter> m_resourceAwsEc2InstanceIpV4Addresses;
    bool m_resourceAwsEc2InstanceIpV4AddressesHasBeenSet = false;

    Aws::Vector<IpFilter> m_resourceAwsEc2InstanceIpV6Addresses;
    bool m_resourceAwsEc2InstanceIpV6AddressesHasBeenSet = false;

    Aws::Vector<StringFilter> m_resourceAwsEc2InstanceKeyName;
    bool m_resourceAwsEc2InstanceKeyNameHasBeenSet = false;

    Aws::Vector<StringFilter> m_resourceAwsEc2InstanceIamInstanceProfileArn;
    bool m_resourceAwsEc2InstanceIamInstanceProfileArnHasBeenSet = false;

    Aws::Vector<StringFilter> m_resourceAwsEc2InstanceVpcId;
    bool m_resourceAwsEc2InstanceVpcIdHasBeenSet = false;

    Aws::Vector<StringFilter> m_resourceAwsEc2InstanceSubnetId;
    bool m_resourceAwsEc2InstanceSubnetIdHasBeenSet = false;

    Aws::Vector<DateFilter> m_resourceAwsEc2InstanceLaunchedAt;
    bool m_resourceAwsEc2InstanceLaunchedAtHasBeenSet = false;

    Aws::Vector<StringFilter> m_resourceAwsS3BucketOwnerId;
    bool m_resourceAwsS3BucketOwnerIdHasBeenSet = false;

    Aws::Vector<StringFilter> m_resourceAwsS3BucketOwnerName;
    bool m_resourceAwsS3BucketOwnerNameHasBeenSet = false;

    Aws::Vector<StringFilter> m_resourceAwsIamAccessKeyPrincipalName;
    bool m_resourceAwsIamAccessKeyPrincipalNameHasBeenSet = false;

    Aws::Vector<StringFilter> m_resourceAwsIamAccessKeyStatus;
    bool m_resourceAwsIamAccessKeyStatusHasBeenSet = false;

    Aws::Vector<DateFilter> m_resourceAwsIamAccessKeyCreatedAt;
    bool m_resourceAwsIamAccessKeyCreatedAtHasBeenSet = false;

    Aws::Vector<StringFilter> m_resourceAwsIamUserUserName;
    bool m_resourceAwsIamUserUserNameHasBeenSet = false;

    Aws::Vector<StringFilter> m_resourceContainerName;
    bool m_resourceContainerNameHasBeenSet = false;

    Aws::Vector<StringFilter> m_resourceContainerImageId;
    bool m_resourceContainerImageIdHasBeenSet = false;

    Aws::Vector<StringFilter> m_resourceContainerImageName;
    bool m_resourceContainerImageNameHasBeenSet = false;

    Aws::Vector<DateFilter> m_resourceContainerLaunchedAt;
    bool m_resourceContainerLaunchedAtHasBeenSet = false;

    Aws::Vector<MapFilter> m_resourceDetailsOther;
    bool m_resourceDetailsOtherHasBeenSet = false;

    Aws::Vector<StringFilter> m_complianceStatus;
    bool m_complianceStatusHasBeenSet = false;

    Aws::Vector<StringFilter> m_verificationState;
    bool m_verificationStateHasBeenSet = false;

    Aws::Vector<StringFilter> m_workflowState;
    bool m_workflowStateHasBeenSet = false;

    Aws::Vector<StringFilter> m_workflowStatus;
    bool m_workflowStatusHasBeenSet = false;

    Aws::Vector<StringFilter> m_recordState;
    bool m_recordStateHasBeenSet = false;

    Aws::Vector<StringFilter> m_relatedFindingsProductArn;
    bool m_relatedFindingsProductArnHasBeenSet = false;

    Aws::Vector<StringFilter> m_relatedFindingsId;
    bool m_relatedFindingsIdHasBeenSet = false;

    Aws::Vector<StringFilter> m_noteText;
    bool m_noteTextHasBeenSet = false;

    Aws::Vector<DateFilter> m_noteUpdatedAt;
    bool m_noteUpdatedAtHasBeenSet = false;

    Aws::Vector<StringFilter> m_noteUpdatedBy;
    bool m_noteUpdatedByHasBeenSet = false;

    Aws::Vector<NumberFilter> m_findingProviderFieldsConfidence;
    bool m_findingProviderFieldsConfidenceHasBeenSet = false;

    Aws::Vector<NumberFilter> m_findingProviderFieldsCriticality;
    bool m_findingProviderFieldsCriticalityHasBeenSet = false;

    Aws::Vector<StringFilter> m_findingProviderFieldsRelatedFindingsId;
    bool m_findingProviderFieldsRelatedFindingsIdHasBeenSet = false;

    Aws::Vector<StringFilter> m_findingProviderFieldsRelatedFindingsProductArn;
    bool m_findingProviderFieldsRelatedFindingsProductArnHasBeenSet = false;

    Aws::Vector<StringFilter> m_findingProviderFieldsSeverityLabel;
    bool m_findingProviderFieldsSeverityLabelHasBeenSet = false;

    Aws::Vector<StringFilter> m_findingProviderFieldsSeverityOriginal;
    bool m_findingProviderFieldsSeverityOriginalHasBeenSet = false;

    Aws::Vector<StringFilter> m_findingProviderFieldsTypes;
    bool m_findingProviderFieldsTypesHasBeenSet = false;

    Aws::Vector<BooleanFilter> m_sample;
    bool m_sampleHasBeenSet = false;

    Aws::Vector<StringFilter> m_complianceSecurityControlId;
    bool m_complianceSecurityControlIdHasBeenSet = false;

    Aws::Vector<StringFilter> m_complianceAssociatedStandardsId;
    bool m_complianceAssociatedStandardsIdHasBeenSet = false;

    Aws::Vector<StringFilter> m_vulnerabilitiesExploitAvailable;
    bool m_vulnerabilitiesExploitAvailableHasBeenSet = false;

    Aws::Vector<StringFilter> m_vulnerabilitiesFixAvailable;
    bool m_vulnerabilitiesFixAvailableHasBeenSet = false;

    Aws::Vector<StringFilter> m_complianceSecurityControlParametersName;
    bool m_complianceSecurityControlParametersNameHasBeenSet = false;

    Aws::Vector<StringFilter> m_complianceSecurityControlParametersValue;
    bool m_complianceSecurityControlParametersValueHasBeenSet = false;

    Aws::Vector<StringFilter> m_awsAccountName;
    bool m_awsAccountNameHasBeenSet = false;

    Aws::Vector<StringFilter> m_resourceApplicationName;
    bool m_resourceApplicationNameHasBeenSet = false;

    Aws::Vector<StringFilter> m_resourceApplicationArn;
    bool m_resourceApplicationArnHasBeenSet = false;
  };

} // namespace Model
} // namespace SecurityHub
} // namespace Aws
