// SPDX-FileCopyrightText: 2024 - 2027 UnionTech Software Technology Co., Ltd.
// SPDX-License-Identifier: GPL-3.0-or-later
import QtQuick 2.15
import QtQuick.Window 2.15
import QtQuick.Controls 2.15
import QtQuick.Layouts 1.15

import org.deepin.dtk 1.0
import org.deepin.dtk.style 1.0 as DStyle

import org.deepin.dcc 1.0

Item {
    id: root
    property real oldX: 180

    Item {
        id: leftView
        anchors {
            top: parent.top
            bottom: parent.bottom
            left: parent.left
            right: splitter.left
        }
        SearchBar {
            id: searchEdit
            anchors {
                left: parent.left
                right: parent.right
                top: parent.top
                margins: 10
                topMargin: 50
            }

            model: DccApp.searchModel()
            onClicked: function (model) {
                DccApp.showPage(model.url)
            }
        }
        ListView {
            id: list
            visible: true
            anchors.top: searchEdit.bottom
            anchors.bottom: parent.bottom
            anchors.left: parent.left
            anchors.right: parent.right
            anchors.margins: 10
            currentIndex: dccObj ? dccObj.children.indexOf(dccObj.currentObject) : -1
            activeFocusOnTab: true
            clip: true
            focus: true
            model: DccModel {
                id: dccModel
                root: dccObj
            }
            delegate: ItemDelegate {
                text: model.display
                width: parent ? parent.width : 300
                checked: dccObj.currentObject === model.item
                icon {
                    name: model.item.icon
                    source: model.item.iconSource
                    width: 24
                    height: 24
                }
                content: Rectangle {
                    anchors.right: parent.right
                    anchors.rightMargin: 10
                    visible: model.item.badge !== 0
                    height: 16
                    width: 16
                    radius: 8
                    color: "red"
                }
                hoverEnabled: true
                background: DccItemBackground {
                    separatorVisible: false
                    backgroundType: DccObject.Hover | DccObject.Clickable
                }

                onClicked: {
                    DccApp.showPage(model.item)
                    console.log(model.item.name, model.display, model.item.icon)
                }
            }
        }
    }
    Rectangle {
        anchors {
            top: parent.top
            bottom: parent.bottom
            left: splitter.right
            right: parent.right
        }
        color: palette.window
        RowLayout {
            id: header
            implicitHeight: 50
            anchors {
                left: parent.left
                right: parent.right
            }
            Item {
                implicitWidth: splitter.x < 110 ? 110 - splitter.x : 0
            }
            ToolButton {
                id: breakBut
                icon.name: "arrow_ordinary_left"
                Layout.alignment: Qt.AlignVCenter | Qt.AlignLeft
                Layout.margins: 10
                implicitHeight: 16
                implicitWidth: 16
                enabled: DccApp.activeObject.parentName.length !== 0 && DccApp.activeObject.parentName !== "root"
                onClicked: DccApp.toBack()
            }

            Crumb {
                implicitHeight: parent.implicitHeight
                implicitWidth: 160
                Layout.fillWidth: true
                Layout.leftMargin: 40
                Layout.rightMargin: 200
                model: DccApp.navModel()
                onClicked: function (model) {
                    DccApp.showPage(model.url)
                }
            }
        }
        StackView {
            id: rightView
            clip: true
            anchors {
                top: header.bottom
                bottom: parent.bottom
                left: parent.left
                right: parent.right
            }
        }
    }
    RowLayout {
        height: 50
        implicitWidth: 100
        ToolButton {
            property real oldSplitterX: 180
            icon.name: "sidebar"
            implicitHeight: 16
            implicitWidth: 16
            Layout.alignment: Qt.AlignVCenter | Qt.AlignLeft
            Layout.leftMargin: 60
            onClicked: {
                if (splitter.x < 110) {
                    var newX = oldSplitterX
                    if (root.width - newX > 520) {
                        splitter.x = newX
                    } else if (root.width - 180 > 520) {
                        splitter.x = 180
                    } else if (root.width - 110 > 520) {
                        splitter.x = 110
                    } else {
                        let dx = 630 - root.width
                        DccApp.mainWindow().x -= dx
                        DccApp.mainWindow().width = 630
                        splitter.x = 110
                    }
                } else {
                    oldSplitterX = splitter.x
                    splitter.x = 0
                    root.oldX = 0
                }
            }
        }
    }
    Rectangle {
        id: splitter
        implicitWidth: 1
        x: 180
        height: root.height
        color: palette.light // "#B9DEFB"
    }
    MouseArea {
        x: splitter.x - 2
        width: 5
        anchors.top: parent.top
        anchors.bottom: parent.bottom
        cursorShape: Qt.SizeHorCursor
        onPositionChanged: function (mouse) {
            var newX = mouse.x + splitter.x
            if (newX >= 0 && newX < root.width - splitter.width) {
                if (root.width - newX < 520) {
                    return
                }
                if (newX < 110) {
                    newX = 0
                }

                splitter.x = newX
                root.oldX = newX
            }
        }
    }

    onWidthChanged: {
        var newX = width - 510
        if (width - splitter.x < 510) {
            if (newX < 0) {
                return
            }
            if (newX < 110) {
                newX = 0
            }
            splitter.x = newX
        } else if (splitter.x < oldX) {

            newX = width - 510
            if (newX < 0) {
                return
            }
            if (newX < 110) {
                newX = 0
            }
            if (newX > oldX) {
                newX = oldX
            }

            splitter.x = newX
        }
    }
    Component {
        id: rightLayout
        DccRightView {}
    }

    function updateRightView() {
        var activeObj = DccApp.activeObject
        if (activeObj === dccObj) {
            return
        }
        if (activeObj.page === null) {
            activeObj.page = rightLayout
        }
        rightView.replace(activeObj.getSectionItem(rightView))
    }
    Connections {
        target: DccApp
        function onActiveObjectChanged() {
            updateRightView()
        }
    }
    Component.onCompleted: updateRightView()
}
